<?php

class RequestForQuotesController extends LoggedUserController
{

    public function __construct($action)
    {
        parent::__construct($action);
        if (UserAuthentication::isGuestUserLogged()) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('account'));
        }
    }

    public function form(int $selProdId)
    {
        if (1 > $selProdId) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }
        $loggedUserId =  UserAuthentication::getLoggedUserId();
        if ($loggedUserId == SellerProduct::getAttributesById($selProdId, 'selprod_user_id')) {
            $this->renderJsonError(Labels::getLabel("LBL_You_can_not_Submit_RFQ_For_Your_Own_Products", $this->siteLangId));
        }
        $parentId = FatApp::getPostedData('parent_id', FatUtility::VAR_INT, 0);
        $services = FatApp::getPostedData('service', FatUtility::VAR_INT, []);
        $servicesArr = [];
        if (!empty($services)) {
            $servicesArr = $this->getServiceArrWithData(array_keys($services));
        }
        if ($parentId > 0) {
            $rfqObj = new RequestForQuote($parentId);
            $servicesArr = $rfqObj->getAttachedServices($this->siteLangId);
        }
        
            /* fetch requested product[ */
        $prodSrch = new ProductSearch($this->siteLangId);
        $prodSrch->setDefinedCriteria(0, 0, array(), false);
        $prodSrch->joinProductToCategory();
        $prodSrch->joinShopSpecifics();
        $prodSrch->joinProductSpecifics();
        $prodSrch->joinSellerProductSpecifics();
        $prodSrch->joinSellerSubscription();
        $prodSrch->addSubscriptionValidCondition();
        $prodSrch->doNotCalculateRecords();
        $prodSrch->addCondition('selprod_id', '=', $selProdId);
        $prodSrch->addCondition('selprod_deleted', '=', AppConstants::NO);
        $prodSrch->doNotLimitRecords();
        $prodSrch->addMultipleFields(
                array('prodcat_name','prodcat_id','selprod_id','shop_user_id','selprod_price', 'special_price_found', 'theprice','shop_country_l.country_name as shop_country_name', 'shop_state_l.state_name as shop_state_name','user_is_verified as seller_verified','user_is_founding_member as is_founding_member', 'shop_is_buy_now as is_buy_now','selprod_user_id','shop_id', 'shop_name','product_id', 'selprod_type','selprod_document_required', 'product_identifier', 'COALESCE(product_name,product_identifier) as product_name', 'product_seller_id', 'COALESCE(selprod_title, product_name, product_identifier) as selprod_title')
        );

        $productRs = $prodSrch->getResultSet();
        $product = FatApp::getDb()->fetch($productRs);
        /* ] */
        if (!$product) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }

        $addresses = UserAddress::getUserAddresses(UserAuthentication::getLoggedUserId(), $this->siteLangId);
        $frm = $this->getForm($servicesArr);
        $frm->fill(array('selprod_id' => $selProdId, 'parent_id' => $parentId));

		$currentPlanData = OrderSubscription::getUserCurrentActivePlanDetails($this->siteLangId, $product['shop_user_id'], array('spackage_id'));
		$currentActivePackageId = FatUtility::int($currentPlanData['spackage_id']);
		$this->set('subscription_package_id' , $currentActivePackageId);
        if ($currentActivePackageId > 0) {
            $packageMemberLabels = SellerPackages::getPackageMemberLabels($this->siteLangId);
            $this->set('subscription_package' , (isset($packageMemberLabels[$currentActivePackageId])) ? $packageMemberLabels[$currentActivePackageId] : '');
        }
		$this->set('shop_rating' ,  SelProdRating::getSellerRating($product['shop_user_id']));


		$addressFrm = $this->getUserAddressForm($this->siteLangId);
		$this->set('addressFrm', $addressFrm);
		$this->set('sellerStats', RequestForQuote::getSellerStates($product['shop_user_id']));
		$this->set('stateId', 0);
        $this->set('frm', $frm);
        $this->set('services', $servicesArr);
        $this->set('addresses', $addresses);
        $this->set('productData', $product);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function formForService(int $selProdId)
    {
        if (1 > $selProdId) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }

        $loggedUserId =  UserAuthentication::getLoggedUserId();
        if ($loggedUserId == SellerProduct::getAttributesById($selProdId, 'selprod_user_id')) {
            $this->renderJsonError(Labels::getLabel("LBL_You_can_not_Submit_RFQ_For_Your_Own_Products", $this->siteLangId));
        }

        $parentId = FatApp::getPostedData('parent_id', FatUtility::VAR_INT, 0);
        $srch = new ServiceSearch($this->siteLangId);
        $srch->joinSellers();
        $srch->joinSellerSubscription();
        $srch->joinShops($this->siteLangId);
        $srch->joinServiceCategory();
        $srch->addMultipleFields(
                array('prodcat_name','prodcat_id','selprod_id','shop_user_id','selprod_price as theprice','country_name as shop_country_name', 'shop_state_l.state_name as shop_state_name','user_is_verified as seller_verified','user_is_founding_member as is_founding_member', 'shop_is_buy_now as is_buy_now','selprod_user_id','shop_id', 'shop_name', 'selprod_type','selprod_document_required','COALESCE(selprod_title) as selprod_title')
        );		
        $srch->addCondition('selprod_id', '=', $selProdId);
        $productRs = $srch->getResultSet();
        $product = FatApp::getDb()->fetch($productRs);
        if (!$product) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }
        $frm = $this->getForm();
        $frm->fill(array('selprod_id' => $selProdId, 'parent_id' => $parentId));
        $addresses = UserAddress::getUserAddresses(UserAuthentication::getLoggedUserId(), $this->siteLangId);
		
		
		$currentPlanData = OrderSubscription::getUserCurrentActivePlanDetails($this->siteLangId, $product['shop_user_id'], array('spackage_id'));
		$currentActivePackageId = FatUtility::int($currentPlanData['spackage_id']);
		$this->set('subscription_package_id' , $currentActivePackageId);
        if ($currentActivePackageId > 0) {
            $packageMemberLabels = SellerPackages::getPackageMemberLabels($this->siteLangId);
            $this->set('subscription_package' , (isset($packageMemberLabels[$currentActivePackageId])) ? $packageMemberLabels[$currentActivePackageId] : '');
        }
		$this->set('shop_rating' ,  SelProdRating::getSellerRating($product['shop_user_id']));		
		
        $this->set('frm', $frm);
		$addressFrm = $this->getUserAddressForm($this->siteLangId);
		$this->set('addressFrm', $addressFrm);		
		$this->set('sellerStats', RequestForQuote::getSellerStates($product['shop_user_id']));
		 $this->set('stateId', 0);
        $this->set('addresses', $addresses);
        $this->set('productData', $product);
        $this->set('siteLangId', $this->siteLangId);
        $this->set('is_service', true);
        $this->_template->render(false, false, 'request-for-quotes/form.php');
    }

    private function getForm(array $services = [])
    {
        $frm = new Form('rfqFrm');
        $frm->addTextBox(Labels::getLabel('LBL_Required_capacity', $this->siteLangId), 'rfq_required_capacity');
        $frm->addRequiredField(Labels::getLabel("LBL_Required_Quantity", $this->siteLangId), 'rfq_quantity')->requirements()->setRange(1, 9999999999);
        $frm->addTextBox(Labels::getLabel("LBL_Delivery_duration", $this->siteLangId), 'rfq_delivery_time', '', array('readonly' => 'readonly', 'class' => 'delivery-date-picker--js'))->requirements()->setRequired();
        $frm->addTextarea(Labels::getLabel('LBL_Comment_for_Seller', $this->siteLangId), 'rfq_comments')->requirements()->setRequired();
        if (!empty($services)) {
            foreach ($services as $serviceId => $val) {
                $frm->addTextBox(Labels::getLabel('LBL_Required_capacity', $this->siteLangId), 'rfq_required_capacity_service[' . $serviceId . ']');
                $frm->addRequiredField(Labels::getLabel("LBL_Required_Quantity", $this->siteLangId), 'rfq_quantity_service[' . $serviceId . ']')->requirements()->setRange(1, 9999999999);
                $frm->addHiddenField('', 'service_group_id[' . $serviceId . ']');
            }
        }

        $frm->addHiddenField('', 'selprod_id', 0);
        $frm->addHiddenField('', 'group_id', 0);
        $frm->addHiddenField('', 'parent_id', 0);
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Submit_request', $this->siteLangId));

        return $frm;
    }

    public function setup()
    {
        $quantity = FatApp::getPostedData('rfq_quantity', FatUtility::VAR_INT, 0);
        $addressId = FatApp::getPostedData('rfq_address_id', FatUtility::VAR_INT, 0);
        $parentId = FatApp::getPostedData('parent_id', FatUtility::VAR_INT, 0);
        $servicesCapacities = FatApp::getPostedData('rfq_required_capacity_service', FatUtility::VAR_STRING, []);
        $servicesQuantities = FatApp::getPostedData('rfq_quantity_service', FatUtility::VAR_INT, []);
        $servicesDocGrpIds = FatApp::getPostedData('service_group_id', FatUtility::VAR_INT, []);


        

        $frm = $this->getForm();
        $postedData = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $postedData) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }

        $loggedUserId = UserAuthentication::getLoggedUserId();

        if ($loggedUserId == SellerProduct::getAttributesById($postedData['selprod_id'], 'selprod_user_id')) {
            $this->renderJsonError(Labels::getLabel("LBL_You_can_not_Submit_RFQ_For_Your_Own_Products", $this->siteLangId));
        }

        if (!User::isBuyer()) {
            $this->renderJsonError(Labels::getLabel("LBL_You_Must_Logged_in_as_buyer_for_Submit_RFQ", $this->siteLangId));
        }

        

        if (1 > $quantity) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Quantity", $this->siteLangId));

        }

        if (0 < $parentId) {
            $reqForQuote = new RequestForQuote($parentId);
            if (empty($reqForQuote->getRequestDetail(UserAuthentication::getLoggedUserId()))) {
                $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request_for_Re-Quote", $this->siteLangId));
            }
        }
        if (empty($addressId)) {
            $this->renderJsonError(Labels::getLabel("LBL_Please_select_delivery_address", $this->siteLangId));
        }

        $addressDetail = UserAddress::getUserAddresses(UserAuthentication::getLoggedUserId(), $this->siteLangId, 0, $addressId);
        if (empty($addressId) && empty($addressDetail)) {
            $this->renderJsonError(Labels::getLabel("LBL_Please_add_delivery_address", $this->siteLangId));
        }

        $data = array(
            'rfq_quantity' => $quantity,
            'rfq_user_id' => UserAuthentication::getLoggedUserId(),
            'rfq_selprod_id' => $postedData['selprod_id'],
            'rfq_capacity' => $postedData['rfq_required_capacity'],
            'rfq_delivery_time' => $postedData['rfq_delivery_time'],
            'rfq_parent_id' => $parentId,
            'rfq_comments' => $postedData['rfq_comments'],
            'rfq_name' => $addressDetail['ua_name'],
            'rfq_shipping_address1' => $addressDetail['ua_address1'],
            'rfq_shipping_address2' => $addressDetail['ua_address2'],
            'rfq_shipping_city' => $addressDetail['ua_city'],
            'rfq_shipping_state' => $addressDetail['state_name'],
            'rfq_shipping_country' => $addressDetail['country_name'],
            'rfq_shipping_zipcode' => $addressDetail['ua_zip'],
            'rfq_shipping_phone' => $addressDetail['ua_phone'],
            'rfq_shipping_state_id' => $addressDetail['ua_state_id'],
            'rfq_shipping_country_id' => $addressDetail['ua_country_id'],
            'rfq_added_on' => date('Y-m-d H:i:s'),
            'rfq_status' => RequestForQuote::REQUEST_INPROGRESS,
        );

        $reqForQuote = new RequestForQuote();
        $reqForQuote->assignValues($data);
        if (!$reqForQuote->save()) {
            $this->renderJsonError(Labels::getLabel("LBL_something_went_wrong_Please_contact_with_administrator", $this->siteLangId));
        }

        $rfqId = $reqForQuote->getMainTableRecordId();
        if (!empty($servicesQuantities)) {
            $this->updateSericesWithProduct($rfqId, $servicesQuantities, $servicesCapacities, $servicesDocGrpIds);
        }

        $groupId = FatApp::getPostedData('group_id', FatUtility::VAR_INT, 0);
        if (0 < $groupId) {
            $shiftRfqDocuments = $this->shiftRfqDocuments($rfqId, $groupId);
            if ($shiftRfqDocuments === true) {
                $db = FatApp::getDb();
                if (!$db->deleteRecords(AttachedFile::DB_TBL_TEMP, array('smt' => 'afile_record_subid = ? ', 'vals' => array($groupId)))) {
                    $this->renderJsonError($db->getError());
                }
            }
        }

        if ($parentId > 0) {
            $reqForQuote = new RequestForQuote($parentId);
            if (false == $reqForQuote->updateStatus(RequestForQuote::REQUEST_RE_QUOTED)) {
                $this->renderJsonError($reqForQuote->getError());
            }
        }

        /* [ SEND EMAIL NOTIFICATION FOR NEW RFQ */
        $emailHandler = new EmailHandler();
        if (!$emailHandler->newRfqNotificationSeller(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $rfqId)) {
            $this->renderJsonSuccess(Labels::getLabel("LBL_Request_submitted_successfully._Error_While_Sending_Email_For_Same", $this->siteLangId));
        }
        /* ] */

        $this->renderJsonSuccess(Labels::getLabel("LBL_Request_submitted_successfully", $this->siteLangId));
    }

    private function updateSericesWithProduct(int $rfqId, array $servicesQuantities, array $servicesCapacities, array $servicesDocGrpIds = []): bool
    {
        if (empty($servicesQuantities) || 1 > $rfqId) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }
        $rfqObj = new RequestForQuote();
        foreach ($servicesQuantities as $serviceId => $quantity) {
            $dataToUpdate = [
                'rfqattser_rfq_id' => $rfqId,
                'rfqattser_selprod_id' => $serviceId,
                'rfqattser_quantity' => $quantity,
                'rfqattser_required_capacity' => (isset($servicesCapacities[$serviceId])) ? $servicesCapacities[$serviceId] : "",
            ];
            if (!$rfqObj->saveServiceWithRfq($dataToUpdate)) {
                $this->renderJsonError($rfqObj->getError());
            }
        }
        if (!empty($servicesDocGrpIds)) {
            $grpSerIds = array_flip($servicesDocGrpIds);
            $criteria = array(
                'afile_record_id' => UserAuthentication::getLoggedUserId(),
                'afile_type' => AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER,
                'afile_record_subid' => array_keys($grpSerIds),
            );
            $attachedFiles = AttachedFile::getTempImagesWithCriteria($criteria);
            if (empty($attachedFiles)) {
                $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            }
            foreach ($attachedFiles as $attachedFile) {
                $serviceId = (isset($grpSerIds[$attachedFile['afile_record_subid']])) ? $grpSerIds[$attachedFile['afile_record_subid']] : 0;
                $attachedFile['afile_record_id'] = $rfqId;
                $attachedFile['afile_record_subid'] = $serviceId;
                unset($attachedFile['afile_id']);
                $record = new AttachedFile();
                $record->assignValues($attachedFile);
                if (!$record->save()) {
                    $this->renderJsonError($record->getError());
                }
            }
        }

        return true;
    }

    private function shiftRfqDocuments(int $rfqId, int $groupId)
    {
        $criteria = array(
            'afile_record_id' => UserAuthentication::getLoggedUserId(),
            'afile_type' => AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER,
            'afile_record_subid' => $groupId,
        );

        $attachedFiles = AttachedFile::getTempImagesWithCriteria($criteria);
        if (empty($attachedFiles)) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }

        foreach ($attachedFiles as $attachedFile) {
            $attachedFile['afile_record_id'] = $rfqId;
            $attachedFile['afile_record_subid'] = 0;
            unset($attachedFile['afile_id']);

            $record = new AttachedFile();
            $record->assignValues($attachedFile);
            if (!$record->save()) {
                $this->renderJsonError($record->getError());
            }
        }

        return true;
    }

    public function listing()
    {
        $frmSearch = $this->searchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    private function searchForm()
    {
        $frm = new Form('frmSearchQuotesRequests');
        $frm->addTextBox('', 'keyword');
        $frm->addSelectBox('', 'prod_type', SellerProduct::selProdType($this->siteLangId), '', array(), Labels::getLabel('LBL_Type', $this->siteLangId));
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $this->siteLangId));
        $frm->addHiddenField('', 'page');
        $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $this->siteLangId), array('onclick' => 'clearSearch();'));
        return $frm;
    }

    public function search()
    {
        $frmSearch = $this->searchForm();
        $post = $frmSearch->getFormDataFromArray(FatApp::getPostedData());

        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->addMultipleFields(array('count(rfq_id) as requestd_count', 'max(rfq_added_on) as latestadded', 'selprod_title', 'rfq_selprod_id', 'IFNULL(unread.unread_count, 0) as unread_count'));
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srch->addCondition('rfq_parent_id', '=', AppConstants::NO);
        $srch->addGroupBy('rfq_selprod_id');
        $srch->setPageSize($pagesize);
        $srch->setPageNumber($page);

        $srchUnreadCount = new RequestForQuoteSearch();
        $srchUnreadCount->doNotCalculateRecords();
        $srchUnreadCount->doNotLimitRecords();
        $srchUnreadCount->addCondition('rfq_marked_read', '=', AppConstants::NO);
        $srch->addCondition('rfq_parent_id', '=', AppConstants::NO);
        $srchUnreadCount->addGroupBy('rfq_selprod_id');
        $srchUnreadCount->addMultipleFields(array('count(rfq_id) as unread_count', "rfq_selprod_id as unread_selprod_id"));
        $srchUnreadCountQuery = $srchUnreadCount->getQuery();
        $srch->joinTable('(' . $srchUnreadCountQuery . ')', 'LEFT OUTER JOIN', 'unread.unread_selprod_id = rfq_selprod_id', 'unread');
        if ($keyword = FatApp::getPostedData('keyword')) {
            $srch->addCondition('selprod_title', 'like', "%$keyword%");
        }

        if (isset($post['prod_type']) && intval($post['prod_type']) != '') {
            $srch->addCondition('selprod_type', '=', intval($post['prod_type']));
        }
        $srch->addOrder('latestadded', 'DESC');
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);

        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set("siteLangId", $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function productQuotes(int $selProdId)
    {
        $checkValidProd = $this->validateSellerProduct($selProdId);
        if ($checkValidProd === false) {
            FatUtility::exitWithErrorCode(404);
        }

        $frmSearch = $this->searchProdQuotesForm($this->siteLangId);
        $frmSearch->fill(array('selprod_id' => $selProdId));
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    private function validateSellerProduct(int $selProdId)
    {
        if (1 > $selProdId) {
            return false;
        }

        $sellerProductRow = SellerProduct::getAttributesById($selProdId);
        if (empty($sellerProductRow)) {
            return false;
        }

        if ($sellerProductRow['selprod_user_id'] != $this->userParentId) {
            return false;
        }

        return true;
    }

    private function searchProdQuotesForm()
    {
        $frm = $this->searchQuoteRequests();
        $frm->addSelectBox('', 'rfq_status', RequestForQuote::statusArray($this->siteLangId), '', array(), Labels::getLabel('LBL_Select_Status', $this->siteLangId));
        $frm->addHiddenField('', 'selprod_id');
        return $frm;
    }

    public function searchReQuotedRequests()
    {
        $frmSearch = $this->reQuotedReqSearchForm();
        $post = $frmSearch->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frmSearch->getValidationErrors()));
        }

        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addMultipleFields(array('rfq_id', 'selprod_title', 'rfq_selprod_id', 'user_name as buyer_name', 'rfq_added_on', 'rfq_status'));
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srch->addCondition('rfq_parent_id', '>', 0);
        $srch->setPageSize($pagesize);
        $srch->setPageNumber($page);

        if ($keyword = FatApp::getPostedData('keyword')) {
            $cnd = $srch->addCondition('selprod_title', 'like', "%$keyword%");
            $cnd->attachCondition('user_name', 'like', "%" . $keyword . "%", 'OR');
            $keyword = str_replace("#", "", $keyword);
            $cnd->attachCondition('rfq_id', 'like', "%" . $keyword . "%", 'OR');
        }

        if (!empty($post['request_from_date'])) {
            $srch->addCondition('rfq_added_on', '>=', $post['request_from_date'] . ' 00:00:00');
        }

        if (!empty($post['request_to_date'])) {
            $srch->addCondition('rfq_added_on', '<=', $post['request_to_date'] . ' 23:59:59');
        }

        if (isset($post['rfq_status']) && $post['rfq_status'] != '') {
            $srch->addCondition('rfq_status', '=', intval($post['rfq_status']));
        }

        if (isset($post['prod_type']) && intval($post['prod_type']) != '') {
            $srch->addCondition('selprod_type', '=', intval($post['prod_type']));
        }

        $srch->addOrder('rfq_added_on', 'DESC');
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);

        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));
        $this->set("siteLangId", $this->siteLangId);
        $this->_template->render(false, false, '/request-for-quotes/search-prod-quotes.php');
    }

    public function searchProdQuotes()
    {
        $frmSearch = $this->searchProdQuotesForm();
        $post = $frmSearch->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frmSearch->getValidationErrors()));
        }
        if (false == $this->validateSellerProduct($post['selprod_id'])) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addMultipleFields(array('rfq_id', 'selprod_title', 'rfq_selprod_id', 'user_name as buyer_name', 'rfq_added_on', 'rfq_status'));
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srch->addCondition('rfq_selprod_id', '=', $post['selprod_id']);
        $srch->addCondition('rfq_parent_id', '=', AppConstants::NO);
        $srch->setPageSize($pagesize);
        $srch->setPageNumber($page);

        if ($keyword = FatApp::getPostedData('keyword')) {
            $cnd = $srch->addCondition('selprod_title', 'like', "%$keyword%");
            $cnd->attachCondition('user_name', 'like', "%" . $keyword . "%", 'OR');
            $keyword = str_replace("#", "", $keyword);
            $cnd->attachCondition('rfq_id', 'like', "%" . $keyword . "%", 'OR');
        }

        if (!empty($post['request_from_date'])) {
            $srch->addCondition('rfq_added_on', '>=', $post['request_from_date'] . ' 00:00:00');
        }

        if (!empty($post['request_to_date'])) {
            $srch->addCondition('rfq_added_on', '<=', $post['request_to_date'] . ' 23:59:59');
        }

        if (isset($post['rfq_status']) && $post['rfq_status'] != '') {
            $srch->addCondition('rfq_status', '=', intval($post['rfq_status']));
        }

        if (isset($post['prod_type']) && intval($post['prod_type']) != '') {
            $srch->addCondition('selprod_type', '=', intval($post['prod_type']));
        }

        $srch->addOrder('rfq_added_on', 'DESC');
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);

        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));

        $this->set("siteLangId", $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function view(int $rfqId)
    {
        if (1 > $rfqId || !$this->userPrivilege->canViewOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            FatUtility::exitWithErrorCode(404);
        }
        $quotedOfferDetail = array();

        $srch = new RequestForQuote($rfqId);
        $record = $srch->getRequestDetail($this->userParentId, $this->siteLangId);
        if (empty($record)) {
            FatUtility::exitWithErrorCode(404);
        }

        if ($record['rfq_status'] > RequestForQuote::REQUEST_INPROGRESS) {
            $counterOffer = new CounterOffer(0, $rfqId);
            $quotedOfferDetail = $counterOffer->getDetailByStatus(RequestForQuote::REQUEST_QUOTED);
        }

        $quotedOfferForm = $this->getQuotedOfferForm();
        $quotedOfferForm->fill(array('counter_offer_rfq_id' => $rfqId));

        if (1 > $record['rfq_marked_read']) {
            $this->markAsRead($rfqId);
        }

        /* [ GET UPLOADED DOCUMENTS SECTION */
        $servicesList = [];
        $sellerAttachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_QUOTED_DOCUMENT, $rfqId, 0, -1);
        $attachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, $rfqId, 0, -1);
        if ($record['selprod_type'] == SellerProduct::PRODUCT_TYPE_PRODUCT) {
            $rfqObj = new RequestForQuote($rfqId);
            $servicesList = $rfqObj->getAttachedServices($this->siteLangId);
            $attachments = CommonHelper::formatAttahmentArr($attachments, 'afile_record_subid');
        }

        $etpl = new FatTemplate('', '');
        $etpl->set('rfqDetail', ['selprod_user_id' => $record['selprod_user_id']]);
        $etpl->set('siteLangId', $this->siteLangId);
        $etpl->set('attachments', $sellerAttachments);
        $etpl->set('hideNoRecordMsg', true);
        $uploadedAttachmentSection = $etpl->render(false, false, 'request-for-quotes/get-uploaded-documents.php', true);
        $this->set('uploadedAttachmentSection', $uploadedAttachmentSection);
        /* ] */

        $srch = new RequestForQuote($rfqId);
        $reQuoteOfferDetail = $srch->getRequestDetailWithParentId(UserAuthentication::getLoggedUserId(), $this->siteLangId);

        $this->set("attachments", $attachments);
        $this->set('servicesList', $servicesList);
        $this->set('rfqData', $record);
        $this->set('quotedOfferDetail', $quotedOfferDetail);
        $this->set('selProdOptions', SellerProduct::getSellerProductOptions($record['rfq_selprod_id'], true, $this->siteLangId));
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));
        $this->set('quotedOfferForm', $quotedOfferForm);
        $this->set('reQuoteOfferDetail', $reQuoteOfferDetail);
        $this->set("siteLangId", $this->siteLangId);
        $this->set("canEdit", $this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true));
        $this->_template->render();
    }

    private function markAsRead(int $rfqId)
    {
        $whr = array('smt' => 'rfq_id = ?', 'vals' => array($rfqId));
        FatApp::getDb()->updateFromArray(RequestForQuote::DB_TBL, array('rfq_marked_read' => 1), $whr);
    }

    private function getQuotedOfferForm()
    {
        $frm = new Form('rfqFrm');

        $fld = $frm->addTextBox(Labels::getLabel('LBL_Product_Total_Cost', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_total_cost');
        $fld->requirements()->setRequired(true);
        $fld->requirements()->setFloatPositive();
        $fld->requirements()->setRange(0, 99999999);

        $fld = $frm->addTextBox(Labels::getLabel('LBL_Shipping_price_cost', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_shipping_cost');
        $fld->requirements()->setRequired(true);
        $fld->requirements()->setFloatPositive();
        $fld->requirements()->setRange(0, 99999999);

        $fldValid = $frm->addTextBox(Labels::getLabel('LBL_Quote_Valid_for_Days', $this->siteLangId), 'quote_validity');
        $fldValid->requirements()->setRequired(true);
        $fldValid->requirements()->setIntPositive();
        $fldValid->requirements()->setRange(1, 99999999);

        $frm->addTextBox(Labels::getLabel("LBL_Delivery_duration", $this->siteLangId), 'counter_offer_delivery_time', '', array('readonly' => 'readonly'));

        $frm->addTextarea(Labels::getLabel('LBL_Comments_for_Buyer', $this->siteLangId), 'counter_offer_comment');
        $frm->addFileUpload(Labels::getLabel('LBL_Upload_document', $this->siteLangId), 'counter_offer_document');

        $frm->addHiddenField('', 'counter_offer_rfq_id');
        $frm->addHiddenField('', 'counter_offer_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Submit', $this->siteLangId));

        return $frm;
    }

    public function uploadDocumentForSeller()
    {
        $groupId = FatApp::getPostedData('group_id', FatUtility::VAR_INT, 0);
        $selprodId = FatApp::getPostedData('selprod_id', FatUtility::VAR_INT, 0);
        if ($groupId > 0) {
            $criteria = array(
                'afile_record_id' => UserAuthentication::getLoggedUserId(),
                'afile_type' => AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER,
                'afile_record_subid' => $groupId,
                'afile_record_subid_2' => $selprodId,
            );
            $attachedFiles = AttachedFile::getTempImagesWithCriteria($criteria);
            if (empty($attachedFiles)) {
                Message::addErrorMessage(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
                FatUtility::dieJsonError(Message::getHtml());
            }
        } else {
            $this->deletePrevUploadedDocuments($selprodId);
            $groupId = rand(1000, 9999);
        }

        if (empty($_FILES) || empty($_FILES['rfq_document']) || !is_uploaded_file($_FILES['rfq_document']['tmp_name'])) {
            Message::addErrorMessage(Labels::getLabel('MSG_Please_select_a_file', $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $fileHandlerObj = new AttachedFile();
        if (!$res = $fileHandlerObj->saveAttachment($_FILES['rfq_document']['tmp_name'], AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, UserAuthentication::getLoggedUserId(), $groupId, $_FILES['rfq_document']['name'], -1, false, 0, 0, 0, true, $selprodId)) {
            Message::addErrorMessage($fileHandlerObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        $json = array(
            'status' => true,
            'msg' => Labels::getLabel('MSG_File_Uploaded_Successfully.', $this->siteLangId),
            'group_id' => $groupId
        );
        FatUtility::dieJsonSuccess($json);
    }

    public function uploadDocument()
    {
        if (!$this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);
        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $rfqId);
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);

        if (empty($record)) {
            Message::addErrorMessage(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        if (empty($_FILES) || empty($_FILES['counter_offer_document']) || !is_uploaded_file($_FILES['counter_offer_document']['tmp_name'])) {
            Message::addErrorMessage(Labels::getLabel('MSG_Please_select_a_file', $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $fileHandlerObj = new AttachedFile();
        if (!$res = $fileHandlerObj->saveAttachment($_FILES['counter_offer_document']['tmp_name'], AttachedFile::FILETYPE_QUOTED_DOCUMENT, $rfqId, 0, $_FILES['counter_offer_document']['name'], -1)) {
            Message::addErrorMessage($fileHandlerObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        Message::addMessage(Labels::getLabel('MSG_File_Uploaded_Successfully.', $this->siteLangId));
        FatUtility::dieJsonSuccess(Message::getHtml());
    }

    public function getUploadedDocuments()
    {
        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $rfqId);
        $cond = $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $cond->attachCondition('rfq_user_id', '=', UserAuthentication::getLoggedUserId(), 'OR');
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);

        if (empty($record)) {
            Message::addErrorMessage(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $attachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_QUOTED_DOCUMENT, $rfqId, 0, -1);
        $this->set("rfqDetail", $record);
        $this->set("siteLangId", $this->siteLangId);
        $this->set("attachments", $attachments);
        $this->set("hideNoRecordMsg", false);
        $this->_template->render(false, false);
    }

    public function removeDocument()
    {
        if (!$this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
        }

        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);
        $afileId = FatApp::getPostedData('afile_id', FatUtility::VAR_INT, 0);
        if (1 > $rfqId || 1 > $afileId) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $rfqId);
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);

        if (empty($record)) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
        }

        $db = FatApp::getDb();
        if (!$db->deleteRecords(AttachedFile::DB_TBL, array('smt' => 'afile_id = ? AND afile_record_id = ? ', 'vals' => array($afileId, $rfqId)))) {
            $this->renderJsonError($db->getError());
        }

        $this->renderJsonSuccess(Labels::getLabel('MSG_File_Removed_Successfully.', $this->siteLangId));
    }

    public function setupOffer()
    {
        $frm = $this->getQuotedOfferForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (!$this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
        }

        if (false === $this->validateRequest($post['counter_offer_rfq_id'])) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Requrest", $this->siteLangId));
        }

        $coff = new CounterOffer(0, $post['counter_offer_rfq_id']);
        $data = $coff->getFinalOfferByRfqId();
        if (!empty($data) && $data['counter_offer_by'] == UserAuthentication::getLoggedUserId()) {
            $this->renderJsonError(Labels::getLabel("MSG_You_Already_Placed_An_Counter_Offer!", $this->siteLangId));
        }

        $defaultData = array(
            'counter_offer_by' => UserAuthentication::getLoggedUserId(),
            'counter_offer_added_on' => date('Y-m-d H:i:s'),
            'counter_offer_status' => RequestForQuote::REQUEST_QUOTED,
        );
        $dataToSave = array_merge($post, $defaultData);

        $record = new CounterOffer($post['counter_offer_id']);
        $record->assignValues($dataToSave);
        if (!$record->save()) {
            $this->renderJsonError($record->getError());
        }

        $dataToUpdate = array(
            'rfq_status' => RequestForQuote::REQUEST_QUOTED,
            'rfq_quote_validity' => $post['quote_validity']
        );
        $record = new RequestForQuote($post['counter_offer_rfq_id']);
        $record->assignValues($dataToUpdate);
        if (!$record->save()) {
            $this->renderJsonError($record->getError());
        }

        /* [ OFFER SUBMISSION EMAIL NOTIFICATION */
        $emailHandler = new EmailHandler();
        if (!$emailHandler->newRfqOfferNotification(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $post['counter_offer_rfq_id'])) {
            $this->renderJsonSuccess(Labels::getLabel('MSG_Offer_submitted_Successfully._But_email_was_not_sent', $this->siteLangId));
        }
        /* ] */
        $this->renderJsonSuccess(Labels::getLabel('MSG_Offer_submitted_Successfully.', $this->siteLangId));
    }

    private function validateRequest(int $rfqId): bool
    {
        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $rfqId);
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);
        if (empty($record)) {
            return false;
        }
        return true;
    }

    public function reQuotedRequests()
    {
        $frmSearch = $this->reQuotedReqSearchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    public function quotedRequests()
    {
        $frmSearch = $this->quotedReqSearchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    private function reQuotedReqSearchForm()
    {
        $frm = $this->searchQuoteRequests();
        $statusArr = RequestForQuote::statusArray($this->siteLangId);
        unset($statusArr[RequestForQuote::REQUEST_RE_QUOTED]);
        $frm->addSelectBox('', 'rfq_status', $statusArr, '', array(), Labels::getLabel('LBL_Select_Status', $this->siteLangId));
        return $frm;
    }

    private function quotedReqSearchForm()
    {
        $frm = $this->searchQuoteRequests();
        $frm->addSelectBox('', 'rfq_status', RequestForQuote::statusArray($this->siteLangId), '', array(), Labels::getLabel('LBL_Select_Status', $this->siteLangId));
        return $frm;
    }

    public function searchBuyerQuotes()
    {
        $postedData = FatApp::getPostedData();
        if (isset($postedData['requote']) && $postedData['requote'] == AppConstants::YES) {
            $frmSearch = $this->reQuotedReqSearchForm();
        } else {
            $frmSearch = $this->quotedReqSearchForm();
        }

        $post = $frmSearch->getFormDataFromArray($postedData);
        if (false === $post) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);

        $srch = $this->searchQuotes($post);
        $srch->addCondition('rfq_user_id', '=', UserAuthentication::getLoggedUserId());

        if (isset($postedData['requote']) && $postedData['requote'] == AppConstants::YES) {
            $srch->addCondition('rfq_parent_id', '>', 0);
        } else {
            $srch->addCondition('rfq_parent_id', '=', AppConstants::NO);
        }
        $srch->setPageNumber($page);
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);
        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));

        $this->set("siteLangId", $this->siteLangId);
        $this->_template->render(false, false);
    }

    private function searchQuotes($post)
    {
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->addMultipleFields(array('rfq_id', 'selprod_title', 'rfq_quote_validity', 'rfq_selprod_id', 'rfq_added_on', 'rfq_status', 'rfq_quantity', 'rfq_capacity', 'selprod_type'));
        $srch->setPageSize($pagesize);

        $keyword = isset($post['keyword']) ? $post['keyword'] : '';
        if ('' != $keyword) {
            $cnd = $srch->addCondition('selprod_title', 'like', "%$keyword%");
            $keyword = str_replace("#", "", $keyword);
            $cnd->attachCondition('rfq_id', 'like', "%" . $keyword . "%", 'OR');
        }

        if (!empty($post['request_from_date'])) {
            $srch->addCondition('rfq_added_on', '>=', $post['request_from_date'] . ' 00:00:00');
        }

        if (!empty($post['request_to_date'])) {
            $srch->addCondition('rfq_added_on', '<=', $post['request_to_date'] . ' 23:59:59');
        }

        if (isset($post['rfq_status']) && $post['rfq_status'] != '') {
            $srch->addCondition('rfq_status', '=', intval($post['rfq_status']));
        }

        if (isset($post['prod_type']) && intval($post['prod_type']) != '') {
            $srch->addCondition('selprod_type', '=', intval($post['prod_type']));
        }

        $srch->addOrder('rfq_added_on', 'DESC');
        return $srch;
    }

    public function requestView(int $rfqId)
    {
        if (1 > $rfqId) {
            FatUtility::exitWithErrorCode(404);
        }

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinWithProduct();
        $srch->joinForSeller();
        $srch->joinShops();
        $srch->addMultipleFields(array('rfq.*', 'selprod_type', 'shop_is_buy_now', 'selprod_updated_on', 'shop_hide_price_for_guests', 'shop_hide_price_for_loginusers',  'rfq_quote_validity', 'selprod_price', 'selprod_title', 'product_updated_on', 'selprod_product_id', 'selprod_id', 'IF(selprod_stock > 0, 1, 0) AS in_stock',));
        $srch->addCondition('rfq_id', '=', $rfqId);
        $srch->addCondition('rfq_user_id', '=', UserAuthentication::getLoggedUserId());
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);

        if (empty($record)) {
            FatUtility::exitWithErrorCode(404);
        }
        $counterOffer = new CounterOffer(0, $rfqId);
        $quotedOfferDetail = $counterOffer->getDetailByStatus(RequestForQuote::REQUEST_QUOTED);

        $srch = new RequestForQuote($rfqId);
        $reQuoteOfferDetail = $srch->getRequestDetailWithParentId(UserAuthentication::getLoggedUserId(), $this->siteLangId);

        $attachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, $rfqId, 0, -1);
        $servicesList = [];

        if ($record['selprod_type'] == SellerProduct::PRODUCT_TYPE_SERVICE) {
            $rfqObj = new RequestForQuote($rfqId);
            $servicesList = $rfqObj->getAttachedServices($this->siteLangId);
            $attachments = CommonHelper::formatAttahmentArr($attachments, 'afile_record_subid');
        }

        $this->set('rfqData', $record);
        $this->set('reQuoteOfferDetail', $reQuoteOfferDetail);
        $this->set('quotedOfferDetail', $quotedOfferDetail);
        $this->set("attachments", $attachments);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));
        $this->set("siteLangId", $this->siteLangId);
        $this->set("servicesList", $servicesList);
        $this->_template->render();
    }

    private function deletePrevUploadedDocuments(int $selprodId = 0)
    {
        $db = FatApp::getDb();
        $smt = 'afile_type = ? AND afile_record_id = ? AND afile_record_subid_2 = ?';
        $smtValues = array(AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, UserAuthentication::getLoggedUserId(), $selprodId);
        if (!$db->deleteRecords(AttachedFile::DB_TBL_TEMP, array('smt' => $smt, 'vals' => $smtValues))) {
            $this->error = $db->getError();
            return false;
        }
        return true;
    }

    public function getTempUploadedDocuments()
    {
        $groupId = FatApp::getPostedData('group_id', FatUtility::VAR_INT, 0);
        $criteria = array(
            'afile_record_id' => UserAuthentication::getLoggedUserId(),
            'afile_type' => AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER,
            'afile_record_subid' => $groupId,
        );
        $attachedFiles = AttachedFile::getTempImagesWithCriteria($criteria);
        if (empty($attachedFiles)) {
            Message::addErrorMessage(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $this->set("records", $attachedFiles);
        $this->set("siteLangId", $this->siteLangId);
        $this->set("serviceId", FatApp::getPostedData('service_id', FatUtility::VAR_INT, 0));
        $this->_template->render(false, false);
    }

    public function removeTempDocument()
    {
        $afileId = FatApp::getPostedData('afile_id', FatUtility::VAR_INT, 0);
        if (1 > $afileId) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Request", $this->siteLangId));
        }

        $criteria = array(
            'afile_record_id' => UserAuthentication::getLoggedUserId(),
            'afile_type' => AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER,
            'afile_id' => $afileId,
        );
        $attachedFiles = AttachedFile::getTempImagesWithCriteria($criteria);
        if (empty($attachedFiles)) {
            $this->renderJsonError(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
        }


        $db = FatApp::getDb();
        if (!$db->deleteRecords(AttachedFile::DB_TBL_TEMP, array('smt' => 'afile_id = ? AND afile_record_id = ? ', 'vals' => array($afileId, UserAuthentication::getLoggedUserId())))) {
            $this->renderJsonError($db->getError());
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_File_Removed_Successfully.', $this->siteLangId));
    }

    public function acceptedOffers()
    {
        if (!$this->userPrivilege->canViewOfferManagement(UserAuthentication::getLoggedUserId(), true) || (!User::canAccessSupplierDashboard() || !User::isSellerVerified($this->userParentId))) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
        }

        $frmSearch = $this->acceptedReqSearchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    public function searchAcceptedQuotes()
    {
        $frmSearch = $this->acceptedReqSearchForm();
        $post = $frmSearch->getFormDataFromArray(FatApp::getPostedData());

        if (false === $post) {
            $this->renderJsonError(current($frmSearch->getValidationErrors()));
        }
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $srch = $this->searchQuotes($post);
        $srch->joinTable(Orders::DB_TBL, 'LEFT OUTER JOIN', 'order_rfq_id = rfq_id', 'o');
        $srch->joinTable(Invoice::DB_TBL, 'LEFT OUTER JOIN', 'o.order_id = invoice.invoice_order_id', 'invoice');
        $srch->addFld(['o.order_is_paid', 'invoice.*', 'o.order_id']);
        $srch->addCondition('selprod_user_id', '=', $this->userParentId);

        $cnd = $srch->addCondition('rfq_status', '=', RequestForQuote::REQUEST_ACCEPTED_BY_BUYER);
        $cnd->attachCondition('rfq_status', '=', RequestForQuote::REQUEST_APPROVED);

        $srch->setPageNumber($page);
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);
        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));

        $this->set("siteLangId", $this->siteLangId);
        $this->set("canEdit", $this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true));
        $this->_template->render(false, false);
    }

    private function acceptedReqSearchForm()
    {
        $frm = $this->searchQuoteRequests();
        $acceptedStatusArr = array(
            RequestForQuote::REQUEST_ACCEPTED_BY_BUYER => Labels::getLabel('LBL_Accepted_by_buyer', $this->siteLangId),
            RequestForQuote::REQUEST_APPROVED => Labels::getLabel('LBL_Accepted_&_Confirmed', $this->siteLangId),
        );
        $frm->addSelectBox('', 'rfq_status', $acceptedStatusArr, '', array(), Labels::getLabel('LBL_Select_Status', $this->siteLangId));
        return $frm;
    }

    public function reQuotedOffers()
    {
        $frmSearch = $this->reQuotedReqSearchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    private function searchQuoteRequests()
    {
        $frm = new Form('frmSearchQuotesRequests');
        $frm->addTextBox('', 'keyword');
        $frm->addDateField('', 'request_from_date', '', array('readonly' => 'readonly'));
        $frm->addDateField('', 'request_to_date', '', array('readonly' => 'readonly'));
        $frm->addSelectBox('', 'prod_type', SellerProduct::selProdType($this->siteLangId), '', array(), Labels::getLabel('LBL_Type', $this->siteLangId));
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $this->siteLangId));
        $frm->addHiddenField('', 'page');
        $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $this->siteLangId), array('onclick' => 'clearSearch();'));
        return $frm;
    }

    private function getMessageSearchForm($langId)
    {
        $frm = new Form('frmMessageSrch');
        $frm->addTextBox('', 'keyword');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $langId));
        $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $langId), array('onclick' => 'clearSearch();'));
        $frm->addHiddenField('', 'page');
        return $frm;
    }

    public function messageSearch()
    {
        $userId = UserAuthentication::getLoggedUserId();
        $userImgUpdatedOn = User::getAttributesById($userId, 'user_updated_on');
        $uploadedTime = AttachedFile::setTimeParam($userImgUpdatedOn);

        $frm = $this->getMessageSearchForm($this->siteLangId);

        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pagesize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);

        $parentAndTheirChildIds = User::getParentAndTheirChildIds($this->userParentId, false, true);

        $srch = new MessageSearch();
        $srch->joinThreadLastMessage();
        $srch->joinMessagePostedFromUser(true, $this->siteLangId);
        $srch->joinMessagePostedToUser(true, $this->siteLangId);
        $srch->joinThreadStartedByUser();
        $srch->addMultipleFields(array('tth.*',
            'ttm.message_id', 'ttm.message_text', 'ttm.message_date', 'ttm.message_is_unread',
            'ttm.message_to', 'IFNULL(tfrs_l.shop_name, tfrs.shop_identifier) as message_from_shop_name',
            'tfrs.shop_id as message_from_shop_id', 'tftos.shop_id as message_to_shop_id',
            'IFNULL(tftos_l.shop_name, tftos.shop_identifier) as message_to_shop_name'));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_type', '=', Thread::THREAD_TYPE_RFQ);
        $cnd = $srch->addCondition('ttm.message_from', 'IN', $parentAndTheirChildIds);
        $cnd->attachCondition('ttm.message_to', 'IN', $parentAndTheirChildIds, 'OR');
        $srch->addOrder('message_id', 'DESC');
        $srch->addGroupBy('ttm.message_thread_id');

        if ($post['keyword'] != '') {
            $cnd = $srch->addCondition('tth.thread_subject', 'like', "%" . $post['keyword'] . "%");
            $cnd->attachCondition('tfr.user_name', 'like', "%" . $post['keyword'] . "%", 'OR');
            $cnd->attachCondition('tfr_c.credential_username', 'like', "%" . $post['keyword'] . "%", 'OR');
        }
        $page = (empty($page) || $page <= 0) ? 1 : $page;
        $page = FatUtility::int($page);
        $srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs);

        // echo "<pre>";
        // print_r($records);
        // die();

        if (API_CALL) {
            $message_records = array();
            foreach ($records as $mkey => $mval) {
                $profile_images_arr = array(
                    "message_from_profile_url" => FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($mval['message_from_user_id'], 'thumb', 1)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg'),
                    "message_to_profile_url" => FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($mval['message_to_user_id'], 'thumb', 1)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg'),
                    "message_timestamp" => strtotime($mval['message_date'])
                );
                $message_records[] = array_merge($mval, $profile_images_arr);
            }
            $records = $message_records;
        }

        /* CommonHelper::printArray($records); die; */
        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('loggedUserId', $userId);
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('parentAndTheirChildIds', $parentAndTheirChildIds);
        $this->set('postedData', $post);

        if (API_CALL) {
            $this->_template->render();
        }
        $this->_template->render(false, false);
    }

    public function rfqMessage($rfq_id)
    {
        $messageId = 0;
        $rfq_id = FatUtility::int($rfq_id);
        $userId = UserAuthentication::getLoggedUserId();
        if (1 > $rfq_id || !$this->userPrivilege->canViewOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            $message = Labels::getLabel('MSG_INVALID_ACCESS', $this->siteLangId);
            if (API_CALL) {
                FatUtility::dieJsonError($message);
            }
            Message::addErrorMessage($message);
            CommonHelper::redirectUserReferer();
        }

        $reqForQuote = new RequestForQuote($rfq_id);
        $rfqDetail = $reqForQuote->getRequestDetail(UserAuthentication::getLoggedUserId());
        if (empty($rfqDetail)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request_for_Re-Quote", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
        

        if ($userId == $rfqDetail['rfq_user_id']) {
            $message_to = $rfqDetail['selprod_user_id'];
        } else {
            $message_to = $rfqDetail['rfq_user_id'];
        }

        $threadData = Thread::getAttributesByRfqId($rfq_id);
        $threadId = 0;
        if ($threadData != false) {
            $threadId = $threadData['thread_id'];
        }

       
        $srch = new MessageSearch();

        $srch->joinThreadMessage();
        $srch->joinMessagePostedFromUser();
        $srch->joinMessagePostedToUser();
        $srch->joinThreadStartedByUser();

        if (isset($threadData['thread_type']) && $threadData['thread_type'] == Thread::THREAD_TYPE_RFQ) {
            $srch->joinRfqProducts($this->siteLangId);
        }

        $parentAndThierChildIds = User::getParentAndTheirChildIds($this->userParentId, false, true);

        $srch->joinOrderProducts();
        $srch->joinOrderProductStatus();
        $srch->addMultipleFields(array('tth.*', 'top.op_invoice_number'));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_id', '=', $threadId);
        if ($messageId) {
            $srch->addCondition('ttm.message_id', '=', $messageId);
        }

        $cnd = $srch->addCondition('ttm.message_from', 'IN', $parentAndThierChildIds);
        $cnd->attachCondition('ttm.message_to', 'IN', $parentAndThierChildIds, 'OR');
        $rs = $srch->getResultSet();
        $threadDetails = FatApp::getDb()->fetch($rs);
      


        if (false === API_CALL) {
            $frmSrch = $this->getMsgSearchForm($this->siteLangId);
            $frmSrch->fill(array('thread_id' => $threadId));
            $frm = $this->sendMessageForm($this->siteLangId);
            $frm->fill(array('rfq_id' => $rfq_id, 'message_thread_id' => $threadId, 'message_id' => $messageId, 'message_to_id' => $message_to));
        }


        $threadObj = new Thread($threadId);
        if (!$threadObj->markMessageReadFromUserArr($threadId, $parentAndThierChildIds)) {
            if (API_CALL) {
                $this->renderJsonError($threadObj->getError());
            }
            Message::addErrorMessage($threadObj->getError());
            CommonHelper::redirectUserReferer();
        }

        if (false === API_CALL) {
            $this->set('frmSrch', $frmSrch);
            $this->set('frm', $frm);
        }
        $this->set('canEditMessages', $this->userPrivilege->canEditMessages(UserAuthentication::getLoggedUserId(), true));
        $this->set('threadDetails', $threadDetails);
        $this->set('threadTypeArr', Thread::getThreadTypeArr($this->siteLangId));
        $this->set('loggedUserId', $userId);
        $this->set('loggedUserName', ucfirst(UserAuthentication::getLoggedUserAttribute('user_name')));
        $this->_template->render();
    }

    private function getMsgSearchForm($langId)
    {
        $frm = new Form('frmMessageSrch');
        $frm->addHiddenField('', 'page');
        $frm->addHiddenField('', 'thread_id');
        return $frm;
    }

    private function sendMessageForm($langId)
    {
        $frm = new Form('frmSendMessage');
        $frm->addTextarea(Labels::getLabel('LBL_Comments', $langId), 'message_text', '')->requirements()->setRequired(true);
        $frm->addHiddenField('', 'rfq_id');
        $frm->addHiddenField('', 'message_thread_id');
        $frm->addHiddenField('', 'message_id');
        $frm->addHiddenField('', 'message_to_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Send', $langId));
        return $frm;
    }

    public function threadMessageSearch()
    {
        $this->userPrivilege->canViewMessages(UserAuthentication::getLoggedUserId());
        $post = FatApp::getPostedData();
        $threadId = empty($post['thread_id']) ? 0 : FatUtility::int($post['thread_id']);

        if (1 > $threadId) {
            $this->_template->render(false, false, 'json-success.php', true, false);
        }

        if (API_CALL) {
            $threadObj = new Thread($threadId);
            if (!$threadObj->markUserMessageRead($threadId, UserAuthentication::getLoggedUserId())) {
                $this->renderJsonError($threadObj->getError());
            }
        }

        $allowedUserIds = User::getParentAndTheirChildIds($this->userParentId, false, true);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pagesize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);

        $srch = new MessageSearch();
        $srch->joinThreadMessage();
        $srch->joinMessagePostedFromUser(true, $this->siteLangId);
        $srch->joinMessagePostedToUser(true, $this->siteLangId);
        $srch->joinThreadStartedByUser();
        $srch->addMultipleFields(array(
            'tth.*', 'ttm.message_id', 'ttm.message_text', 'ttm.message_date', 'ttm.message_is_unread',
            'IFNULL(tfrs_l.shop_name, tfrs.shop_identifier) as message_from_shop_name', 'tfrs.shop_id as message_from_shop_id',
            'tftos.shop_id as message_to_shop_id', 'IFNULL(tftos_l.shop_name, tftos.shop_identifier) as message_to_shop_name'));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_id', '=', $threadId);
        $cnd = $srch->addCondition('ttm.message_from', 'in', $allowedUserIds);
        $cnd->attachCondition('ttm.message_to', 'in', $allowedUserIds, 'OR');
        $srch->addOrder('message_id', 'DESC');
        $srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs, 'message_id');
        ksort($records);
        $this->set("arrListing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);

        $startRecord = ($page - 1) * $pagesize + 1;
        $endRecord = $pagesize;
        $totalRecords = $srch->recordCount();
        if ($totalRecords < $endRecord) {
            $endRecord = $totalRecords;
        }

        $this->set('totalRecords', $totalRecords);
        $this->set('startRecord', $startRecord);
        $this->set('endRecord', $endRecord);
        $this->set('records', $records);

        if (API_CALL) {
            $this->_template->render();
        }

        $this->set('loadMoreBtnHtml', $this->_template->render(false, false, '_partial/load-previous-btn.php', true));
        $this->set('html', $this->_template->render(false, false, 'account/thread-message-search.php', true, false));
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    public function sendMessage()
    {
        $userId = UserAuthentication::getLoggedUserId();

        $frm = $this->sendMessageForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        if (false === $post) {
            if (API_CALL) {
                LibHelper::dieJsonError(current($frm->getValidationErrors()));
            }
            Message::addErrorMessage(current($frm->getValidationErrors()));
            FatUtility::dieWithError(Message::getHtml());
        }

        $threadId = FatUtility::int($post['message_thread_id']);
        $rfqId = FatUtility::int($post['rfq_id']);
        $message_to_id = FatUtility::int($post['message_to_id']);

        $srch = new MessageSearch();
        $srch->addCondition('tth.thread_id', '=', $threadId);
        $rs = $srch->getResultSet();
        $threadDetails = FatApp::getDb()->fetch($rs);
        if (empty($threadDetails)) {
            $threadObj = new Thread();
            $threadDataToSave = array(
                'thread_subject' => Thread::RFQ_SUBJECT,
                'thread_started_by' => $userId,
                'thread_start_date' => date('Y-m-d H:i:s'),
                'thread_type' => Thread::THREAD_TYPE_RFQ,
                'thread_record_id' => $rfqId
            );
            $threadObj->assignValues($threadDataToSave);
            if (!$threadObj->save()) {
                $message = Labels::getLabel($threadObj->getError(), $this->siteLangId);
                if (API_CALL) {
                    FatUtility::dieJsonError($message);
                }
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }
            $threadId = $threadObj->getMainTableRecordId();
        } else {
            $threadId = $threadDetails['thread_id'];
        }

        $data = array(
            'message_thread_id' => $threadId,
            'message_from' => $userId,
            'message_to' => $message_to_id,
            'message_text' => $post['message_text'],
            'message_date' => date('Y-m-d H:i:s'),
            'message_is_unread' => 1
        );

        $tObj = new Thread();

        if (!$insertId = $tObj->addThreadMessages($data)) {
            $this->renderJsonError($tObj->getError());
        }
        $emailObj = new EmailHandler();
        $emailObj->SendMessageNotification($insertId, $this->siteLangId);
        

        $this->set('threadId', $threadId);
        $this->set('messageId', $insertId);
        $this->set('msg', Labels::getLabel('MSG_Message_Submitted_Successfully!', $this->siteLangId));
        if (API_CALL) {
            $this->set('messageDetail', $data);
            $this->_template->render();
        }
        $this->_template->render(false, false, 'json-success.php');
    }

    public function getServiceArrWithData(array $services): array
    {
        if (empty($services)) {
            return [];
        }
        $srch = SellerProduct::getSearchObject($this->siteLangId);
        $srch->addCondition('selprod_id', 'in', $services);
        $srch->addCondition('selprod_active', '=', AppConstants::YES);
        $srch->addCondition('selprod_deleted', '=', AppConstants::NO);
        $srch->addCondition('selprod_type', '=', SellerProduct::PRODUCT_TYPE_SERVICE);
        $srch->addMultipleFields(['selprod_id', 'selprod_document_required', 'selprod_title']);
        $rs = $srch->getResultSet();
        return FatApp::getDb()->fetchAll($rs, 'selprod_id');
    }

}
