<?php

class InvoicesController extends LoggedUserController
{

    public function __construct($action)
    {
        parent::__construct($action);
        if (UserAuthentication::isGuestUserLogged()) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('account'));
        }
    }

    public function create(string $orderId)
    {
        if (!$this->userPrivilege->canEditInvoices(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Seller'));
            exit;
        }
        $orderDetail = $this->getOrderDetails($orderId);
        if (empty($orderDetail) || (isset($orderDetail['invoice_status']) && $orderDetail['invoice_status'] == Invoice::INVOICE_IS_SHARED_WITH_BUYER)) {
            if (User::canAccessSupplierDashboard() || User::isSellerVerified($this->userParentId)) {
                //Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('seller', 'sales'));
            } else {
                Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('home'));
            }
        }
        $rfqStatus = RequestForQuote::getRfqStatus($orderDetail['order_rfq_id']);
        if ($rfqStatus['rfq_status'] == RequestForQuote::REQUEST_QUOTE_VALIDITY) {
            Message::addErrorMessage(Labels::getLabel("LBL_Quote_Expired", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Seller'));
            exit;
        }

        $totalAmountToPay = $orderDetail['order_net_amount'] - $orderDetail['total_paid_amount'];
        $frm = $this->getForm($orderDetail['order_is_paid'], $totalAmountToPay);
        $invoiceData = Invoice::getInvoiceDetailsByOrderId($orderId);

        if (empty($invoiceData)) {
            $invoiceData['invoice_order_id'] = $orderId;
            $coObj = new CounterOffer(0, $orderDetail['order_rfq_id']);

            $offerDetails = $coObj->getFinalOfferByRfqId(true);
            $invoiceData['invoice_delivery_duration'] = $offerDetails['counter_offer_delivery_time'];
        }

        if ($orderDetail['order_is_paid'] == Orders::ORDER_IS_PARTIAL_PAID) {
            $invoiceData['invoice_payment_type'] = Orders::PAYMENT_TYPE_FULL;
            $invoiceData['invoice_payment_amount'] = $totalAmountToPay;
        }

        $frm->fill($invoiceData);
        $this->set('frm', $frm);
        $this->set('siteLangId', $this->siteLangId);
        $this->set('orderId', $orderId);
        $this->set('orderDetails', $orderDetail);
        $this->_template->render();
    }

    public function generateInvoice()
    {
        if (!$this->userPrivilege->canEditInvoices(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
        /* NEED TO ADD CONDITION FOR VALID SELLER */
        $orderId = FatApp::getPostedData('invoice_order_id', FatUtility::VAR_STRING, '');
        if ($orderId == '') {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $orderDetails = $this->getOrderDetails($orderId);

        if (empty($orderDetails)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $totalAmountToPay = $orderDetails['order_net_amount'] - $orderDetails['total_paid_amount'];
        $frm = $this->getForm($orderDetails['order_is_paid'], $totalAmountToPay);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        if ($post === false) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        if ( ($post['invoice_expiry_link'] < date("Y-m-d",strtotime('+2 days'))))     {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_expired", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $invoiceId = $post['invoice_id'];
        $post['invoice_status'] = Invoice::INVOICE_IS_PENDING;
        unset($post['invoice_id']);

        $invObj = new Invoice($invoiceId, $orderId);
        $invObj->assignValues($post);
        if (!$invObj->save()) {
            Message::addErrorMessage($invObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        $orderObj = new Orders();
        $orderObj->updateOrderInfo($orderId, array('order_partial_payment' => $post['invoice_payment_amount']));

        /* [ UPDATE INVOICE REQUEST STATUS  */
        $dataToSave = array(
            'inreq_order_id' => $orderId,
            'inreq_status' => InvoiceRequest::INVOICE_REQUEST_COMPLETE,
        );

        $invReqObj = new InvoiceRequest($orderId);
        if (!$invReqObj->saveInvoiceRequest($dataToSave)) {
            Message::addErrorMessage($invReqObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
        /* ] */

        $invoiceId = $invObj->getMainTableRecordId();
        $this->set('invoiceId', $invoiceId);
        $this->set('orderId', $orderId);
        $this->set('msg', Labels::getLabel('MSG_Invoice_Generated_Successfully', $this->siteLangId));
        $this->_template->render(false, false, 'json-success.php');
    }

    public function view($orderId, $print = false)
    {
        if (!$this->userPrivilege->canViewInvoices(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Seller'));
        }

        $this->setInvoiceCommonData($orderId, 2);
        $this->set('print', $print);
        $this->set('action', 'view');
        $this->set('is_buyer', false);
        $this->set('is_seller', true);
        $this->set('canEdit', $this->userPrivilege->canEditInvoices(UserAuthentication::getLoggedUserId(), true));
        if ($print) {
            $this->_template->render(true, true, 'invoices/print-invoice-view.php');
        } else {
            $this->_template->render();
        }
    }

    public function sendInvoice(string $orderId)
    {
        if (!$this->userPrivilege->canEditInvoices(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $srch = new OrderProductSearch($this->siteLangId, true, true);
        $srch->joinOrderUser();
        $srch->joinTable(Invoice::DB_TBL, 'INNER JOIN', 'invoice.invoice_order_id = op.op_order_id', 'invoice');
        $srch->joinSellerProducts();
        $srch->addCondition('op_selprod_user_id', '=', $this->userParentId);
        $srch->addCondition('op_order_id', '=', $orderId);
        $srch->addCondition('order_is_rfq', '=', AppConstants::YES);
        $srch->addMultipleFields(
                array(
                    'order_id', 'user_id', 'credential_email as buyer_email', 'user_name as buyer_name',
                    'selprod_title', 'invoice.*', 'op_shop_name as shop_name', 'order_net_amount as total_amount',
                    'op_shop_owner_name as shop_owner', 'op_shop_id'
                )
        );
        $rs = $srch->getResultSet();
        $orderDetail = FatApp::getDb()->fetch($rs);

        if (empty($orderDetail)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $obj = new Orders();
        $totalPaidAmount = $obj->getOrderPaymentPaid($orderId);
        $orderDetail['total_paid_amount'] = $totalPaidAmount;

        $tpl = new FatTemplate('', '');
        $tpl->set('siteLangId', $this->siteLangId);
        $tpl->set('orderInfo', $orderDetail);
        $invoiceTableHtml = $tpl->render(false, false, '_partial/emails/invoice-email.php', true);
        $data = [
            'shop_logo' => FatCache::getCachedUrl(CommonHelper::generateFileUrl('Image', 'shopEmailLogo', array($orderDetail['op_shop_id'], $this->siteLangId), CONF_WEBROOT_URL), CONF_IMG_CACHE_TIME, '.jpg'),
            'user_id' => $orderDetail['user_id'],
            'user_name' => $orderDetail['buyer_name'],
            'user_email' => $orderDetail['buyer_email'],
            'invoice_url' => CommonHelper::generateFullUrl('Invoices', 'PayInvoice', [$orderId], CONF_WEBROOT_FRONT_URL),
            'invoice_details_table' => $invoiceTableHtml,
            'order_id' => $orderDetail['order_id']
        ];


        /* [ UPDATE INVOICE STATUS TO SHARED */
        $dataToUpdate = array('invoice_status' => invoice::INVOICE_IS_SHARED_WITH_BUYER, 'invoice_added_on' => date('Y-m-d H:i:s'));
        $inObj = new Invoice($orderDetail['invoice_id']);
        $inObj->assignValues($dataToUpdate);
        if (!$inObj->save()) {
            Message::addErrorMessage($inObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
        /* ] */

        $emailHandler = new EmailHandler();
        if (!$emailHandler->sendInvoiceLink(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $data)) {
            Message::addErrorMessage($emailHandler->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }


        $this->set('msg', Labels::getLabel('MSG_Invoice_Mail_Sent_to_Buyer', $this->siteLangId));
        $this->_template->render(false, false, 'json-success.php');
    }

    public function payInvoice(string $orderId, $print = false)
    {
        $this->setInvoiceCommonData($orderId, 1);
        $this->set('print', $print);
        $this->set('action', 'payInvoice');
        $this->set('is_buyer', true);
        $this->set('is_seller', false);
        $this->_template->addCss(array('invoices/page-css/view.css'));
        if ($print) {
            $this->_template->render(true, true, 'invoices/print-invoice-view.php');
        } else {
            $this->_template->render(true, true, 'invoices/view.php');
        }
    }

    public function invoiceRequestForm(string $orderId)
    {
        $frm = $this->getRequestForm();
        $frm->fill(['inreq_order_id' => $orderId]);
        $this->set('frm', $frm);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function requestForInvoice()
    {
        $frm = $this->getRequestForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        if ($post === false) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $srch = $this->getOrderSrchObj($post['inreq_order_id']);
        $rs = $srch->getResultSet();
        $orderDetail = FatApp::getDb()->fetch($rs);

        if (empty($orderDetail)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $dataToSave = array(
            'inreq_order_id' => $post['inreq_order_id'],
            'inreq_reason' => $post['inreq_reason'],
            'inreq_user_id' => UserAuthentication::getLoggedUserId(),
            'inreq_status' => InvoiceRequest::INVOICE_REQUEST_PENDING,
            'inreq_added_on_date' => date('Y-m-d H:i:s'),
            'inreq_seller_user_id' => $orderDetail['op_selprod_user_id']
        );

        $invReqObj = new InvoiceRequest($post['inreq_order_id']);
        if (!$invReqObj->saveInvoiceRequest($dataToSave)) {
            Message::addErrorMessage($invReqObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
        /* [ UPDATE INVOICE STATUS */
        $invoiceObj = new Invoice($orderDetail['invoice_id']);
        $invoiceObj->assignValues(['invoice_status' => Invoice::INVOICE_IS_PENDING]);
        if (!$invoiceObj->save()) {
            Message::addErrorMessage($invoiceObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
        /* ] */

        $tpl = new FatTemplate('', '');
        $tpl->set('siteLangId', $this->siteLangId);
        $tpl->set('orderInfo', $orderDetail);
        $invoiceTableHtml = $tpl->render(false, false, '_partial/emails/invoice-email.php', true);

        $data = [
            'shop_logo' => FatCache::getCachedUrl(CommonHelper::generateFileUrl('Image', 'shopEmailLogo', array($orderDetail['op_shop_id'], $this->siteLangId), CONF_WEBROOT_URL), CONF_IMG_CACHE_TIME, '.jpg'),
            'user_id' => $orderDetail['user_id'],
            'user_name' => $orderDetail['user_name'],
            'user_email' => $orderDetail['credential_email'],
            'invoice_url' => CommonHelper::generateFullUrl('Invoices', 'create', [$post['inreq_order_id']], CONF_WEBROOT_FRONT_URL),
            'invoice_details_table' => $invoiceTableHtml,
            'order_id' => $orderDetail['order_id'],
            'seller_id' => $orderDetail['selprod_user_id'],
            'seller_name' => $orderDetail['op_shop_owner_name']. ' - '. $orderDetail['op_shop_name'],
            'product_name' => $orderDetail['op_selprod_title'],
            'seller_email' => $orderDetail['op_shop_owner_email']
        ];

        $emailHandler = new EmailHandler();
        if (!$emailHandler->requestForInvoice(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $data)) {
            Message::addErrorMessage($emailHandler->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        $this->set('msg', Labels::getLabel('MSG_Your_Request_is_Submitted_Successfully', $this->siteLangId));
        $this->_template->render(false, false, 'json-success.php');
    }

    public function invoiceRequests()
    {
        if (!$this->userPrivilege->canViewInvoices(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Access", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Seller'));
        }

        $frmSearch = $this->searchForm($this->siteLangId);
        $this->set("frmSearch", $frmSearch);
        $this->_template->render();
    }

    public function searchRequests()
    {
        $frmSearch = $this->searchForm();
        $post = $frmSearch->getFormDataFromArray(FatApp::getPostedData());

        $page = FatApp::getPostedData('page', FatUtility::VAR_INT, 1);
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE', FatUtility::VAR_INT, 10);

        $srch = InvoiceRequest::getSearchObj();
        $srch->addCondition('inreq_seller_user_id', '=', $this->userParentId);
        $srch->joinTable(User::DB_TBL, 'INNER JOIN', 'user_id = inreq.inreq_user_id AND user_deleted = 0', 'buyer_user');
        $srch->addMultipleFields(array('inreq.*', 'buyer_user.user_name as buyer_name'));

        $srch->setPageSize($pagesize);
        $srch->setPageNumber($page);

        $rs = $srch->getResultSet();
        $requestListing = FatApp::getDB()->fetchAll($rs);

        $this->set('statusArr', InvoiceRequest::getRequestStatusArr($this->siteLangId));
        $this->set('listingData', $requestListing);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);
        $this->set("siteLangId", $this->siteLangId);
        $this->set("canEdit", $this->userPrivilege->canEditInvoices(UserAuthentication::getLoggedUserId(), true));
        $this->_template->render(false, false);
    }

    public function previousInvoice($orderId, $print = false) // SELLER
    {
        $this->setPartialPaidInvoiceCommonData($orderId, 2);
        $this->set('print', $print);
        $this->set('action', 'previousInvoice');
        $this->set('is_buyer', false);
        $this->set('is_seller', true);
        $this->_template->addCss(array('invoices/page-css/view.css'));
        if ($print) {
            $this->_template->render(true, true, 'invoices/print-paid-invoice.php');
        } else {
            $this->_template->render(true, true, 'invoices/paid-invoice.php');
        }
    }

    public function paidInvoice($orderId, $print = false) // == BUYER
    {
        $this->setPartialPaidInvoiceCommonData($orderId, 1);
        $this->set('print', $print);
        $this->set('action', 'paidInvoice');
        $this->set('is_buyer', true);
        $this->set('is_seller', false);
        $this->_template->addCss(array('invoices/page-css/view.css'));
        if ($print) {
            $this->_template->render(true, true, 'invoices/print-paid-invoice.php');
        } else {
            $this->_template->render();
        }
    }

    private function setPartialPaidInvoiceCommonData($orderId, $type)
    {
        $srch = $this->getOrderSrchObj($orderId);
        $srch->addCondition('order_is_paid', 'IN', [/* Orders::ORDER_IS_PAID, */Orders::ORDER_IS_PARTIAL_PAID]);
        $srch->joinTable(Orders::DB_TBL_ORDER_PAYMENTS, 'INNER JOIN', 'order_id = opayment_order_id', 'opayment');
        $srch->addOrder('opayment_id', 'DESC');
        if ($type == 1) {
            $srch->addCondition('order_user_id', '=', UserAuthentication::getLoggedUserId());
        } else {
            $srch->addCondition('op_selprod_user_id', '=', $this->userParentId);
        }
        $rs = $srch->getResultSet();
        $orderDetail = FatApp::getDb()->fetch($rs);

        if (empty($orderDetail)) {

            if ($type == 2) {
                Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_URL", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'listing'));
            } else {
                Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_URL", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'quotedRequests'));
            }
        }

        $orderObj = new Orders();
        $orderDetail['total_paid_amount'] = $orderObj->getOrderPaymentPaid($orderId);

        $address = $orderObj->getOrderAddresses($orderDetail['op_order_id']);
        $orderDetail['billingAddress'] = (isset($address[Orders::BILLING_ADDRESS_TYPE])) ? $address[Orders::BILLING_ADDRESS_TYPE] : array();
        $orderDetail['shippingAddress'] = (isset($address[Orders::SHIPPING_ADDRESS_TYPE])) ? $address[Orders::SHIPPING_ADDRESS_TYPE] : array();
        $taxOptions = json_decode($orderDetail['op_product_tax_options'], true);
        $orderDetail['taxOptions'] = $taxOptions;

        $charges = $orderObj->getOrderProductChargesByOrderId($orderDetail['order_id']);
        $orderDetail['charges'] = $charges[$orderDetail['op_id']];

        $shopAddress = Shop::getShopAddress($orderDetail['op_shop_id'], true, $this->siteLangId);

        $attachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, $orderDetail['order_rfq_id'], 0, -1);
        $rfqObj = new RequestForQuote($orderDetail['order_rfq_id']);
        $this->set('rfqServices', $rfqObj->getAttachedServices($this->siteLangId));
        $this->set('attachments', CommonHelper::formatAttahmentArr($attachments, 'afile_record_subid'));

        $this->set('shopAddress', $shopAddress);
        $this->set('orderDetail', $orderDetail);
        $this->set('orderStatuses', Orders::getOrderProductStatusArr($this->siteLangId));
        $this->set('languages', Language::getAllNames());
        $this->set('yesNoArr', AppConstants::getYesNoArr($this->siteLangId));
        $this->set('paymentTypes', Orders::getPaymentTypeArr($this->siteLangId));
    }

    private function setInvoiceCommonData($orderId, int $type) /* $type 1 = Buyer 2 = Seller */
    {
        $srch = $this->getOrderSrchObj($orderId);
        $srch->addCondition('order_is_paid', 'IN', Orders::getUnpaidStatus());

        if ($type == 1) {
            $srch->addCondition('order_user_id', '=', UserAuthentication::getLoggedUserId());
            $srch->addCondition('invoice_status', '=', Invoice::INVOICE_IS_SHARED_WITH_BUYER);
        } else {
            $srch->addCondition('op_selprod_user_id', '=', $this->userParentId);
            $srch->addCondition('invoice_status', 'IN', [Invoice::INVOICE_IS_SHARED_WITH_BUYER, Invoice::INVOICE_IS_PENDING]);
        }


        $rs = $srch->getResultSet();
        $orderDetail = FatApp::getDb()->fetch($rs);
        if ( ($type == 1 && $orderDetail['invoice_expiry_link'] < date("Y-m-d H:i:s")))     {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_URL_OR_Invoice_is_expired", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'listing'));
        }
        
        $rfqStatus = RequestForQuote::getRfqStatus($orderDetail['order_rfq_id']);
        if ($rfqStatus['rfq_status'] == RequestForQuote::REQUEST_QUOTE_VALIDITY) {
            Message::addErrorMessage(Labels::getLabel("LBL_Quote_Expired", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'quotedRequests'));
            exit;
        }

        if (empty($orderDetail)) {
            if ($type == 2) {
                Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_URL_OR_Invoice_is_already_paid", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'listing'));
            } else {
                Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Invoice_URL_OR_Invoice_is_already_paid", $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'quotedRequests'));
            }
        }

        $orderObj = new Orders();
        $address = $orderObj->getOrderAddresses($orderDetail['op_order_id']);
        $orderDetail['billingAddress'] = (isset($address[Orders::BILLING_ADDRESS_TYPE])) ? $address[Orders::BILLING_ADDRESS_TYPE] : array();
        $orderDetail['shippingAddress'] = (isset($address[Orders::SHIPPING_ADDRESS_TYPE])) ? $address[Orders::SHIPPING_ADDRESS_TYPE] : array();
        $taxOptions = json_decode($orderDetail['op_product_tax_options'], true);
        $orderDetail['taxOptions'] = $taxOptions;

        $invoicePaymentHistory = $orderObj->getOrderPayments(array("order_id" => $orderDetail['order_id']));
        $orderDetail['total_paid_amount'] = 0;
        if (!empty($invoicePaymentHistory)) {
            $orderDetail['total_paid_amount'] = array_sum(array_column($invoicePaymentHistory, 'opayment_amount'));
        }

        $charges = $orderObj->getOrderProductChargesByOrderId($orderDetail['order_id']);
        $orderDetail['charges'] = $charges[$orderDetail['op_id']];

        $shopAddress = Shop::getShopAddress($orderDetail['op_shop_id'], true, $this->siteLangId);
        $this->set('shopAddress', $shopAddress);

        $attachments = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER, $orderDetail['order_rfq_id'], 0, -1);

        $this->set('orderDetail', $orderDetail);
        $rfqObj = new RequestForQuote($orderDetail['order_rfq_id']);
        $this->set('rfqServices', $rfqObj->getAttachedServices($this->siteLangId));
        $this->set('attachments', CommonHelper::formatAttahmentArr($attachments, 'afile_record_subid'));
        $this->set('paymentHistory', $invoicePaymentHistory);
        $this->set('orderStatuses', Orders::getOrderProductStatusArr($this->siteLangId));
        $this->set('languages', Language::getAllNames());
        $this->set('yesNoArr', AppConstants::getYesNoArr($this->siteLangId));
        $this->set('paymentTypes', Orders::getPaymentTypeArr($this->siteLangId));
    }

    private function getOrderSrchObj($orderId)
    {
        $srch = new OrderProductSearch($this->siteLangId, true, true);
        $srch->joinTable(Invoice::DB_TBL, 'INNER JOIN', 'invoice.invoice_order_id = op.op_order_id', 'invoice');
        $srch->joinPaymentMethod();
        $srch->joinSellerProducts();
        $srch->joinOrderUser();
        $srch->joinShippingUsers();
        $srch->joinShippingCharges();
        $srch->addOrderProductCharges();
        $srch->addStatusCondition(unserialize(FatApp::getConfig("CONF_VENDOR_ORDER_STATUS")));
        $srch->addCondition('op_order_id', '=', $orderId);
        $srch->addCondition('order_is_rfq', '=', AppConstants::YES);
        $srch->addCondition('op_status_id', '!=', FatApp::getConfig('CONF_DEFAULT_CANCEL_ORDER_STATUS'));
        return $srch;
    }

    private function getOrderDetails($orderId)
    {
        $srch = new OrderProductSearch($this->siteLangId, true, true);
        $srch->joinTable(Invoice::DB_TBL, 'LEFT OUTER JOIN', 'invoice.invoice_order_id = op.op_order_id', 'invoice');
        $srch->joinSellerProducts();
        $srch->addCondition('op_selprod_user_id', '=', $this->userParentId);
        $srch->addCondition('op_order_id', '=', $orderId);
        $srch->addCondition('order_is_rfq', '=', AppConstants::YES);
        $srch->addCondition('order_is_paid', 'IN', Orders::getUnpaidStatus());
        $srch->addCondition('op_status_id', '!=', FatApp::getConfig('CONF_DEFAULT_CANCEL_ORDER_STATUS'));
        //$srch->addMultipleFields(['order_net_amount', 'order_id', 'order_is_paid', 'IFNULL(invoice_status, 0) as invoice_status', 'order_rfq_id', 'order_tax_charged', 'op_actual_shipping_charges']);

        $rs = $srch->getResultSet();
        $orderData = FatApp::getDb()->fetch($rs);

        if ($orderData == false) {
            return [];
        }

        $orderData['total_paid_amount'] = 0;
        if ($orderData['order_is_paid'] == Orders::ORDER_IS_PARTIAL_PAID) {
            $obj = new Orders();
            $orderData['total_paid_amount'] = $obj->getOrderPaymentPaid($orderId);
        }
        return $orderData;
    }

    private function getForm($orderStatus, $totalAmountToPay)
    {
        $frm = new Form('frmInvoice');
        $paymentTypesArr = Orders::getPaymentTypeArr($this->siteLangId);
        $selectedPaymentType = 0;
        if ($orderStatus == Orders::ORDER_IS_PARTIAL_PAID) {
            unset($paymentTypesArr[Orders::PAYMENT_TYPE_PARTIAL]);
        }

        $fld = $frm->addSelectBox(Labels::getLabel('LBL_Payment_Type', $this->siteLangId), 'invoice_payment_type', $paymentTypesArr, '');
        $fld->requirements()->setRequired(true);

        if ($orderStatus == Orders::ORDER_IS_PARTIAL_PAID) {
            $frm->addHiddenField('', 'invoice_payment_amount', $totalAmountToPay, ['data-totalamount' => $totalAmountToPay]);
        } else {
            $fld = $frm->addTextBox(Labels::getLabel('LBL_Amount', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'invoice_payment_amount', 0, ['data-totalamount' => $totalAmountToPay]);
            $fld->requirements()->setRequired(true);
            $fld->requirements()->setFloatPositive();
            $fld->requirements()->setRange(0.1, $totalAmountToPay);
        }
        /* $fld = $frm->addSelectBox(Labels::getLabel('LBL_Currency', $this->siteLangId), 'invoice_currency', Currency::getCurrencyAssoc($this->siteLangId), '');
          $fld->requirements()->setRequired(true); */

        //$frm->addDateTimeField(Labels::getLabel("LBL_Delivery_Duration", $this->siteLangId), 'invoice_delivery_duration', '', array('readonly' => 'readonly'));
        $frm->addTextBox(Labels::getLabel("LBL_Delivery_Duration", $this->siteLangId), 'invoice_delivery_duration', '', array('readonly' => 'readonly', 'class' => 'delivery-date-picker--js'));
        $fld = $frm->addTextBox(Labels::getLabel("LBL_Expiry_link_valid_till_date", $this->siteLangId), 'invoice_expiry_link', '', array('readonly' => 'readonly', 'class' => 'expiry-date-picker--js'));
		$fld->requirements()->setRequired(true);
        $frm->addTextarea(Labels::getLabel('LBL_Description', $this->siteLangId), 'invoice_description');
        $frm->addHiddenField('', 'invoice_order_id');
        $frm->addHiddenField('', 'invoice_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Generate_Invoice', $this->siteLangId));
        return $frm;
    }

    private function searchForm()
    {
        $frm = new Form('frmSearchQuotesRequests');
        $frm->addTextBox('', 'keyword');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $this->siteLangId));
        $frm->addHiddenField('', 'page', 1);
        $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $this->siteLangId), array('onclick' => 'clearSearch();'));
        return $frm;
    }

    private function getRequestForm()
    {
        $frm = new Form('frmInvoiceRequest');
        $frm->addTextarea(Labels::getLabel('LBL_Reason', $this->siteLangId), 'inreq_reason');
        $frm->addHiddenField('', 'inreq_order_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Submit', $this->siteLangId));
        return $frm;
    }

}
