<?php

class CounterOfferController extends LoggedUserController
{

    public function __construct($action)
    {
        parent::__construct($action);
        if (UserAuthentication::isGuestUserLogged()) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('account'));
        }
    }

    public function listingForBuyer()
    {
        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);
        if (1 > $rfqId) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $rfqId);
        $srch->addCondition('rfq_user_id', '=', UserAuthentication::getLoggedUserId());
        $srchRs = $srch->getResultSet();
        $record = FatApp::getDb()->fetch($srchRs);

        if (empty($record)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $srch = CounterOffer::getSearchObject($rfqId);
        $srch->joinTable(RequestForQuote::DB_TBL, 'INNER JOIN', 'co.counter_offer_rfq_id = rfq.rfq_id', 'rfq');
        $srch->addOrder('counter_offer_added_on', 'DESC');
        $srch->addMultipleFields(array('co.*', 'rfq.rfq_quote_validity', 'rfq.rfq_added_on'));
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);

        $this->set("rfqData", $record);
        $this->set("arr_listing", $records);
        $this->set("siteLangId", $this->siteLangId);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));
        $this->_template->render(false, false);
    }

    public function listingForSeller()
    {
        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);

        $reqForQuote = new RequestForQuote($rfqId);
        $rfqDetail = $reqForQuote->getRequestDetail($this->userParentId);

        if (empty($rfqDetail) || $rfqDetail['selprod_user_id'] != $this->userParentId) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $srch = CounterOffer::getSearchObject($rfqId);
        $srch->addOrder('counter_offer_added_on', 'DESC');
        $srchRs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($srchRs);

        $this->set("rfqData", $rfqDetail);
        $this->set("arr_listing", $records);
        $this->set("siteLangId", $this->siteLangId);
        $this->set('statusArr', RequestForQuote::statusArray($this->siteLangId));
        $this->set('canEdit', $this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true));
        $this->_template->render(false, false);
    }

    public function form()
    {
        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);
        $shippingCost = $this->getShippingRate($rfqId);
        if (!empty($data)) {
            $shippingCost = $data['counter_offer_shipping_cost'];
        }
        $frm = $this->getForm($shippingCost);
        $frm->fill(array('counter_offer_rfq_id' => $rfqId, 'counter_offer_shipping_cost' => $shippingCost));
        $this->set('frm', $frm);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function formForSeller()
    {
        $rfqId = FatApp::getPostedData('rfq_id', FatUtility::VAR_INT, 0);
        $shippingCost = $this->getShippingRate($rfqId);
        $frm = $this->getSellerCounterForm($shippingCost);
        $frm->fill(array('counter_offer_rfq_id' => $rfqId, 'counter_offer_shipping_cost' => $shippingCost));
        $this->set('frm', $frm);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false, '/counter-offer/form.php');
    }

    private function getForm($shippingCost = 0)
    {
        $frm = new Form('counterOfferFrm');

        $fld = $frm->addTextBox(Labels::getLabel('LBL_Offer_Price', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_total_cost');
        $fld->requirements()->setRequired(true);
        $fld->requirements()->setFloatPositive();
        $fld->requirements()->setRange(0.1, 999999);
        $frm->addTextBox(Labels::getLabel('LBL_Shipping_Cost', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_shipping_cost', '', ['readonly' => 'readonly']);
        $frm->addHTML('', 'total_price', '<div ><span>' . Labels::getLabel('LBL_Total_Price', $this->siteLangId) . ' : </span><span class="total_price--js">' . $shippingCost . '</span></div>');
        $frm->addTextarea(Labels::getLabel('LBL_Comment', $this->siteLangId), 'counter_offer_comment');
        $frm->addHiddenField('', 'counter_offer_rfq_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Submit_Offer', $this->siteLangId));

        return $frm;
    }

    private function getSellerCounterForm($shippingCost = 0)
    {
        $frm = new Form('counterOfferFrm');
        $fld = $frm->addTextBox(Labels::getLabel('LBL_Offer_Price', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_total_cost');
        $fld->requirements()->setRequired(true);
        $fld->requirements()->setFloatPositive();
        $fld->requirements()->setRange(0.1, 999999);
        $fld = $frm->addTextBox(Labels::getLabel('LBL_Shipping_price_cost', $this->siteLangId) . ' [' . CommonHelper::getSystemDefaultCurrenyCode() . ']', 'counter_offer_shipping_cost');
        $fld->requirements()->setRequired(true);
        $fld->requirements()->setFloatPositive();
        $fld->requirements()->setRange(0.1, 999999);
        $fldValid = $frm->addTextBox(Labels::getLabel('LBL_Quote_Valid_for_Days', $this->siteLangId), 'quote_validity');
        $fldValid->requirements()->setRequired(true);
        $fldValid->requirements()->setIntPositive();
        $frm->addHTML('', 'total_price', '<div ><span>' . Labels::getLabel('LBL_Total_Price', $this->siteLangId) . ' : </span><span class="total_price--js">' . $shippingCost . '</span></div>');
        $frm->addTextarea(Labels::getLabel('LBL_Comment', $this->siteLangId), 'counter_offer_comment');
        $frm->addHiddenField('', 'counter_offer_rfq_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Submit_Offer', $this->siteLangId));

        return $frm;
    }

    public function setupBuyerCounterOffer()
    {
        $frm = $this->getForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if ($post === false) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }
        if (isset($post['counter_offer_rfq_id'])) {
            $post['counter_offer_shipping_cost'] = $this->getShippingRate($post['counter_offer_rfq_id']);
        }
        $this->setup($post, RequestForQuote::REQUEST_COUNTER_BY_BUYER);
    }

    public function setupSellerCounterOffer()
    {
        $frm = $this->getSellerCounterForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        $this->setup($post, RequestForQuote::REQUEST_COUNTER_BY_SELLER);
    }

    public function setup(array $post, int $status)
    {
        if ($post === false) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }
        $rfqId = intval($post['counter_offer_rfq_id']);
        if (1 > $rfqId) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }
        $defaultData = array(
            'counter_offer_by' => UserAuthentication::getLoggedUserId(),
            'counter_offer_added_on' => date('Y-m-d H:i:s'),
            'counter_offer_status' => $status,
        );
        $dataToSave = array_merge($post, $defaultData);
        $record = new CounterOffer();
        $record->assignValues($dataToSave);
        if (!$record->save()) {
            $this->renderJsonError($record->getError());
        }
        $dataToUpdate = array(
            'rfq_status' => $status
        );

        $record = new RequestForQuote($rfqId);
        $record->assignValues($dataToUpdate);
        if (!$record->save()) {
            $this->renderJsonError($record->getError());
        }

        /* [ OFFER SUBMISSION EMAIL NOTIFICATION */
        $emailHandler = new EmailHandler();
        if (!$emailHandler->newRfqOfferNotification(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $rfqId)) {
            $this->renderJsonSuccess(Labels::getLabel('MSG_Offer_submitted_Successfully._Error_while_sending_email.', $this->siteLangId));
        }
        /* ] */
        $this->renderJsonSuccess(Labels::getLabel('MSG_Offer_submitted_Successfully.', $this->siteLangId));
    }

    public function updateStatusByBuyer()
    {
        $post = FatApp::getPostedData();
        if (1 > $post['rfq_id'] || !RequestForQuote::canBuyerUpdateStatus($post['status'])) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }
        $this->changeStatus($post);
    }

    public function updateStatusBySeller()
    {
        if (!$this->userPrivilege->canEditOfferManagement(UserAuthentication::getLoggedUserId(), true)) {
            Message::addErrorMessage(Labels::getLabel("MSG_Invalid_Access", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        $post = FatApp::getPostedData();
        if (1 > $post['rfq_id'] || !RequestForQuote::canSellerUpdateStatus($post['status'])) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
        $this->changeStatus($post);
    }

    public function changeStatus($post)
    {
        if (1 > $post['rfq_id']) {
            $this->renderJsonError(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
        }

        $srch = new RequestForQuoteSearch();
        $srch->joinWithSellerProduct($this->siteLangId);
        $srch->joinUsers();
        $srch->addCondition('rfq_id', '=', $post['rfq_id']);
        $cond = $srch->addCondition('selprod_user_id', '=', $this->userParentId);
        $cond->attachCondition('rfq_user_id', '=', UserAuthentication::getLoggedUserId(), 'OR');
        $srchRs = $srch->getResultSet();
        $rfqData = FatApp::getDb()->fetch($srchRs);

        if (empty($rfqData)) {
            $this->renderJsonError(Labels::getLabel("MSG_INVALID_ACCESS", $this->siteLangId));
        }
        $dataToUpdate = array(
            'rfq_status' => $post['status']
        );
        $db = FatApp::getDb();
        $db->startTransaction();
        $record = new RequestForQuote($post['rfq_id']);
        $record->assignValues($dataToUpdate);
        if (!$record->save()) {
            $this->renderJsonError($record->getError());
        }
       
        if ($post['status'] == RequestForQuote::REQUEST_APPROVED) {
            if (!$this->addOrderForRfq($post['rfq_id'])) {
                $db->rollbackTransaction();
                $this->renderJsonError($db->getError());
            }
        }
       
        $db->commitTransaction();
        /* [ OFFER SUBMISSION EMAIL NOTIFICATION */
        $emailHandler = new EmailHandler();
        if (!$emailHandler->offerStatusUpdateNotification(FatApp::getConfig('CONF_DEFAULT_SITE_LANG', FatUtility::VAR_INT, 1), $post['rfq_id'])) {
            $this->renderJsonSuccess(Labels::getLabel('MSG_Status_updated_Successfully._Error_while_sending_email', $this->siteLangId));
        }
        /* ] */
        $this->renderJsonSuccess(Labels::getLabel('MSG_Status_updated_Successfully.', $this->siteLangId));
    }

    public function downloadDigitalFile(int $rfqId, int $aFileId, int $fileType, $isPreview = false, $w = 100, $h = 100)
    {
        if (1 > $aFileId || 1 > $rfqId) {
            FatUtility::exitWithErrorCode(404);
        }

        $reqForQuote = new RequestForQuote($rfqId);
        $rfqDetail = $reqForQuote->getRequestDetail($this->userParentId);

        if (empty($rfqDetail)) {
            FatUtility::exitWithErrorCode(404);
        }

        $attachFileRow = AttachedFile::getAttributesById($aFileId);

        /* files path[ */
        $folderName = '';
        switch ($fileType) {
            case AttachedFile::FILETYPE_QUOTED_DOCUMENT:
                $folderName = AttachedFile::FILETYPE_RFQ_DOCUMENT_PATH;
                break;
            case AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER:
                $folderName = AttachedFile::FILETYPE_SERVICE_DOCUMENT_PATH;
                break;
        }
        /* ] */

        if (!file_exists(CONF_UPLOADS_PATH . $folderName . $attachFileRow['afile_physical_path'])) {
            Message::addErrorMessage(Labels::getLabel('LBL_File_not_found', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('RequestForQuotes', 'RequestView', array($rfqId)));
        }

        if ($isPreview) {
            AttachedFile::displayImage($folderName . $attachFileRow['afile_physical_path'], $w, $h);
        } else {
            AttachedFile::downloadAttachment($folderName . $attachFileRow['afile_physical_path'], $attachFileRow['afile_name']);
        }
    }

    private function addOrderForRfq(int $rfqId)
    {
        $rfqDetails = RequestForQuote::getAttributesById($rfqId);
        $coObj = new CounterOffer(0, $rfqId);
        $offerDetails = $coObj->getFinalOfferByRfqId();
        if (empty($rfqDetails) || empty($offerDetails)) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
            return false;
        }

        if (!Orders::validRfqForOrder($rfqId)) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
            return false;
        }

        $rfq = new RequestForQuote();
        $productType = SellerProduct::getAttributesById($rfqDetails['rfq_selprod_id'], 'selprod_type');
        $productInfo = $rfq->getCartProductInfo($rfqDetails['rfq_selprod_id'], $productType);
        $unitPrice = $offerDetails["counter_offer_total_cost"] / $rfqDetails['rfq_quantity'];

        
        $shippingAddress = $billingAddress = array(
            'ua_address1' => $rfqDetails['rfq_shipping_address1'],
            'ua_address2' => $rfqDetails['rfq_shipping_address2'],
            'ua_city' => $rfqDetails['rfq_shipping_city'],
            'ua_state_id' => $rfqDetails['rfq_shipping_state_id'],
            'ua_country_id' => $rfqDetails['rfq_shipping_country_id'],
            'ua_phone' => $rfqDetails['rfq_shipping_phone'],
            'ua_zip' => $rfqDetails['rfq_shipping_zipcode'],
        );

        $extraData = array(
            'billingAddress' => $billingAddress,
            'shippingAddress' => $shippingAddress,
            'shippedBySeller' => 1, /*RFQ order is shipped by seller because shipping cose decided by seller*/
            'shippingCost' => $offerDetails['counter_offer_shipping_cost'],
            'buyerId' => $rfqDetails['rfq_user_id'],
            'productType' => $productType
        );
                    
        $taxObj = new Tax();
		$productId = $productInfo['product_id'];
		if($productType == SellerProduct::PRODUCT_TYPE_SERVICE) {
			$productId = $rfqDetails['rfq_selprod_id'];
		}
 		
        $taxData = $taxObj->calculateTaxRates($productId, $unitPrice, $productInfo['selprod_user_id'], $this->siteLangId, $rfqDetails['rfq_quantity'], $extraData, false);
        $productInfo['tax'] = $taxData['tax'];

        $maxConfiguredCommissionVal = FatApp::getConfig("CONF_MAX_COMMISSION");
        $commissionPercentage = SellerProduct::getProductCommission($rfqDetails['rfq_selprod_id']);
        $commission = MIN(ROUND(($offerDetails["counter_offer_total_cost"] + $offerDetails['counter_offer_shipping_cost'] + $taxData['tax']) * $commissionPercentage / 100, 2), $maxConfiguredCommissionVal);
        $productInfo['commission_percentage'] = $commissionPercentage;
        $productInfo['commission'] = ROUND($commission, 2);

        $orderData = [];
        $orderData['order_id'] = '';
        $orderData['order_user_id'] = $rfqDetails['rfq_user_id'];
        $orderData['order_date_added'] = date('Y-m-d H:i:s');
        $orderData['order_date_updated'] = date('Y-m-d H:i:s');
        $orderData['order_is_rfq'] = AppConstants::YES;
        $orderData['order_rfq_id'] = $rfqId;
        $userAddresses[0] = array(
            'oua_order_id' => '',
            'oua_type' => Orders::BILLING_ADDRESS_TYPE,
            'oua_name' => $rfqDetails['rfq_name'],
            'oua_address1' => $rfqDetails['rfq_shipping_address1'],
            'oua_address2' => $rfqDetails['rfq_shipping_address2'],
            'oua_city' => $rfqDetails['rfq_shipping_city'],
            'oua_state' => $rfqDetails['rfq_shipping_state'],
            'oua_country' => $rfqDetails['rfq_shipping_country'],
            /* 'oua_country_code' => $rfqDetails['country_code'], */
            'oua_phone' => $rfqDetails['rfq_shipping_phone'],
            'oua_zip' => $rfqDetails['rfq_shipping_zipcode'],
        );
        $userAddresses[1] = $userAddresses[0];
        $userAddresses[1]['oua_type'] = Orders::SHIPPING_ADDRESS_TYPE;
        $orderData['userAddresses'] = $userAddresses;

        $orderData['extra'] = array(
            'oextra_order_id' => '',
            'order_ip_address' => $_SERVER['REMOTE_ADDR']
        );

        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $orderData['extra']['order_forwarded_ip'] = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } elseif (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $orderData['extra']['order_forwarded_ip'] = $_SERVER['HTTP_CLIENT_IP'];
        } else {
            $orderData['extra']['order_forwarded_ip'] = '';
        }

        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $orderData['extra']['order_user_agent'] = $_SERVER['HTTP_USER_AGENT'];
        } else {
            $orderData['extra']['order_user_agent'] = '';
        }

        if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $orderData['extra']['order_accept_language'] = $_SERVER['HTTP_ACCEPT_LANGUAGE'];
        } else {
            $orderData['extra']['order_accept_language'] = '';
        }

        $languageRow = Language::getAttributesById($this->siteLangId);
        $orderData['order_language_id'] = $languageRow['language_id'];
        $orderData['order_language_code'] = $languageRow['language_code'];

        $currencyRow = Currency::getAttributesById($this->siteCurrencyId);
        $orderData['order_currency_id'] = $currencyRow['currency_id'];
        $orderData['order_currency_code'] = $currencyRow['currency_code'];
        $orderData['order_currency_value'] = $currencyRow['currency_value'];
        $orderData['order_default_currency_to_cop'] = 1;
        $columbianCurrValue = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
        if (!empty($columbianCurrValue)) {
            $orderData['order_default_currency_to_cop'] = $columbianCurrValue['currency_value'];
        }
        
        $orderData['order_user_comments'] = '';
        $orderData['order_admin_comments'] = '';

        $orderData['order_tax_charged'] = $productInfo['tax'];
        $orderData['order_site_commission'] = $productInfo['commission'];
        $orderData['order_volume_discount_total'] = 0;

        $orderData['order_net_amount'] = $offerDetails["counter_offer_total_cost"] + $offerDetails["counter_offer_shipping_cost"] + $productInfo['tax'];
        $orderData['order_is_wallet_selected'] = 0;
        $orderData['order_wallet_amount_charge'] = 0;
        $orderData['order_type'] = Orders::ORDER_PRODUCT;

        $allLanguages = Language::getAllNames();
        $orderLangData = array();
        $productShippingData['opshipping_by_seller_user_id'] = $productInfo['selprod_user_id'];


        foreach ($allLanguages as $lang_id => $language_name) {
            $orderLangData[$lang_id] = array(
                'orderlang_lang_id' => $lang_id,
                'order_shippingapi_name' => ''
            );
            $langSpecificProductInfo = $rfq->getCartProductLangData($rfqDetails['rfq_selprod_id'], $lang_id, $productType);
            if (!$langSpecificProductInfo) {
                continue;
            }

            $weightUnitsArr = AppConstants::getWeightUnitsArr($lang_id);
            $lengthUnitsArr = AppConstants::getLengthUnitsArr($lang_id);
            $op_selprod_title = ($langSpecificProductInfo['selprod_title'] != '') ? $langSpecificProductInfo['selprod_title'] : '';

            $op_selprod_options = '';
            $productOptionsRows = SellerProduct::getSellerProductOptions($rfqDetails['rfq_selprod_id'], true, $lang_id);
            if (!empty($productOptionsRows)) {
                $optionCounter = 1;
                foreach ($productOptionsRows as $poLang) {
                    $op_selprod_options .= $poLang['option_name'] . ': ' . $poLang['optionvalue_name'];
                    if ($optionCounter != count($productOptionsRows)) {
                        $op_selprod_options .= ' | ';
                    }
                    $optionCounter++;
                }
            }

            $op_products_dimension_unit_name = ($productInfo['product_dimension_unit']) ? $lengthUnitsArr[$productInfo['product_dimension_unit']] : '';
            $op_product_weight_unit_name = ($productInfo['product_weight_unit']) ? $weightUnitsArr[$productInfo['product_weight_unit']] : '';
            $op_product_tax_options = array();

            if (array_key_exists('options', $taxData)) {        
                foreach ($taxData['options'] as $taxStroId => $taxStroName) {
                    $label = Labels::getLabel('LBL_Tax', $lang_id);
                    if (array_key_exists('name', $taxStroName) && $taxStroName['name'] != '') {
                        $label = $taxStroName['name'];
                    }
                    $op_product_tax_options[$label]['name'] = $label;
                    $op_product_tax_options[$label]['value'] = $taxStroName['value'];
                    $op_product_tax_options[$label]['percentageValue'] = $taxStroName['percentageValue'];
                    $op_product_tax_options[$label]['inPercentage'] = $taxStroName['inPercentage'];
                }
            }
            
            $sduration_name = '';
            $shippingDurationTitle = '';
            $productsLangData[$lang_id] = [
                'oplang_lang_id' => $lang_id,
                'op_product_name' => $langSpecificProductInfo['product_name'],
                'op_selprod_title' => $op_selprod_title,
                'op_selprod_options' => $op_selprod_options,
                'op_brand_name' => (isset($langSpecificProductInfo['brand_name']) && !empty($langSpecificProductInfo['brand_name'])) ? $langSpecificProductInfo['brand_name'] : '',
                'op_shop_name' => $langSpecificProductInfo['shop_name'],
                'op_shipping_duration_name' => $sduration_name,
                'op_shipping_durations' => $shippingDurationTitle,
                'op_products_dimension_unit_name' => $op_products_dimension_unit_name,
                'op_product_weight_unit_name' => $op_product_weight_unit_name,
                'op_product_tax_options' => json_encode($op_product_tax_options),
            ];
        }

        $orderData['products'][CART::CART_KEY_PREFIX_PRODUCT . $productInfo['selprod_id']] = [
            'op_selprod_id' => $productInfo['selprod_id'],
            'op_is_batch' => 0,
            'op_type' => $productInfo['selprod_type'],
            'op_selprod_user_id' => $productInfo['selprod_user_id'],
            'op_selprod_code' => $productInfo['selprod_code'],
            'op_qty' => $rfqDetails['rfq_quantity'],
            'op_unit_price' => $unitPrice,
            'op_unit_cost' => ($offerDetails["counter_offer_total_cost"] + $offerDetails["counter_offer_shipping_cost"] + $productInfo['tax']) / $rfqDetails['rfq_quantity'],
            'op_selprod_sku' => $productInfo['selprod_sku'],
            'op_selprod_condition' => $productInfo['selprod_condition'],
            'op_product_model' => $productInfo['product_model'],
            'op_product_type' => $productInfo['product_type'],
            'op_product_length' => $productInfo['product_length'],
            'op_product_width' => $productInfo['product_width'],
            'op_product_height' => $productInfo['product_height'],
            'op_product_dimension_unit' => $productInfo['product_dimension_unit'],
            'op_product_weight' => $productInfo['product_weight'],
            'op_product_weight_unit' => $productInfo['product_weight_unit'],
            'op_shop_id' => $productInfo['shop_id'],
            'op_shop_owner_username' => $productInfo['shop_owner_username'],
            'op_shop_owner_name' => $productInfo['shop_onwer_name'],
            'op_shop_owner_email' => $productInfo['shop_owner_email'],
            'op_shop_owner_phone' => isset($productInfo['shop_owner_phone']) && !empty($productInfo['shop_owner_phone']) ? $productInfo['shop_owner_phone'] : '',
            'op_selprod_max_download_times' => ($productInfo['selprod_max_download_times'] != '-1') ? $rfqDetails['rfq_quantity'] * $productInfo['selprod_max_download_times'] : $productInfo['selprod_max_download_times'],
            'op_selprod_download_validity_in_days' => $productInfo['selprod_download_validity_in_days'],
            'op_sduration_id' => 0,
            'op_commission_charged' => $productInfo['commission'],
            'op_commission_percentage' => $productInfo['commission_percentage'],
            'op_affiliate_commission_percentage' => 0,
            'op_affiliate_commission_charged' => 0,
            'op_status_id' => FatApp::getConfig("CONF_DEFAULT_ORDER_STATUS"),
            'productsLangData' => $productsLangData,
            'productShippingData' => $productShippingData,
            'productShippingLangData' => [],
            'op_free_ship_upto' => 0,
            'op_actual_shipping_charges' => $offerDetails['counter_offer_shipping_cost'],
            'op_tax_collected_by_seller' => FatApp::getConfig("CONF_TAX_COLLECTED_BY_SELLER"),
            'productSpecifics' => [
                'op_selprod_return_age' => $productInfo['return_age'],
                'op_selprod_cancellation_age' => $productInfo['cancellation_age'],
                'op_product_warranty' => $productInfo['product_warranty']
            ]
        ];

        $shippingCost = $offerDetails["counter_offer_shipping_cost"];
        $orderData['prodCharges'][CART::CART_KEY_PREFIX_PRODUCT . $productInfo['selprod_id']] = [
            OrderProduct::CHARGE_TYPE_SHIPPING => ['amount' => $shippingCost],
            OrderProduct::CHARGE_TYPE_TAX => ['amount' => $taxData['tax']],
            OrderProduct::CHARGE_TYPE_DISCOUNT => ['amount' => 0],
            OrderProduct::CHARGE_TYPE_REWARD_POINT_DISCOUNT => ['amount' => 0],
            OrderProduct::CHARGE_TYPE_VOLUME_DISCOUNT => ['amount' => 0],
        ];

        $orderData['orderLangData'] = $orderLangData;
        $orderObj = new Orders();
        if (!$orderObj->addUpdateOrder($orderData, $this->siteLangId)) {
            Message::addErrorMessage($orderObj->getError());
            return false;
        }

        return true;
    }

   

   

    private function getShippingRate(int $rfqId)
    {
        /* [ GET SHIPPING PRICE FROM SELLER OFFER */
        $srch = CounterOffer::getSearchObject($rfqId);
        $srch->addOrder('co.counter_offer_id', 'DESC');
        $srch->addFld('counter_offer_shipping_cost');
        $srchRs = $srch->getResultSet();
        $data = FatApp::getDb()->fetch($srchRs);
        if (empty($data)) {
            return 0;
        }
        return $data['counter_offer_shipping_cost'];
    }

    public function downloadTempDigitalFile(int $aFileId, int $fileType, $isPreview = false, $w = 100, $h = 100)
    {
        if (1 > $aFileId) {
            FatUtility::exitWithErrorCode(404);
        }
        $srch = new SearchBase(AttachedFile::DB_TBL_TEMP, 'aft');
        $srch->addCondition('aft.afile_id', '=', $aFileId);
        $rs = $srch->getResultSet();
        $attachFileRow = FatApp::getDb()->fetch($rs);
        if ($attachFileRow === false) {
            FatUtility::exitWithErrorCode(404);
        }
        /* files path[ */
        $folderName = '';
        switch ($fileType) {
            case AttachedFile::FILETYPE_QUOTED_DOCUMENT:
                $folderName = AttachedFile::FILETYPE_RFQ_DOCUMENT_PATH;
                break;
            case AttachedFile::FILETYPE_SERVICE_DOCUMENTS_FOR_SELLER:
                $folderName = AttachedFile::FILETYPE_SERVICE_DOCUMENT_PATH;
                break;
        }
        /* ] */

        if (!file_exists(CONF_UPLOADS_PATH . $folderName . $attachFileRow['afile_physical_path'])) {
            Message::addErrorMessage(Labels::getLabel('LBL_File_not_found', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('RequestForQuotes', 'RequestView', array($attachFileRow['afile_record_id'])));
        }

        if ($isPreview) {
            AttachedFile::displayImage($folderName . $attachFileRow['afile_physical_path'], $w, $h);
        } else {
            AttachedFile::downloadAttachment($folderName . $attachFileRow['afile_physical_path'], $attachFileRow['afile_name']);
        }
    }

}
